package follows

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strconv"
	"sync"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/common/twitchhttp"
)

const (
	defaultTimingXactName = "follows"
	defaultStatSampleRate = 1.0
	// MaxFollows is the maximum amount of users that a user can follow
	MaxFollows = 2000
)

// Client is a client for the following service
type Client interface {
	GetFollow(ctx context.Context, fromUserID string, targetUserID string, opts *twitchhttp.ReqOpts) (*Follow, error)
	ListFollows(ctx context.Context, fromUserID string, params *ListFollowsParams, opts *twitchhttp.ReqOpts) (*Follows, error)
	ListFollowers(ctx context.Context, fromUserID string, params *ListFollowersParams, opts *twitchhttp.ReqOpts) (*Followers, error)
	CountFollowers(ctx context.Context, userID string, opts *twitchhttp.ReqOpts) (int, error)
	BulkCountFollowers(ctx context.Context, userIDs []string, opts *twitchhttp.ReqOpts) (map[string]int, error)
	Follow(ctx context.Context, fromUserID string, targetUserID string, body *FollowRequestBody, opts *twitchhttp.ReqOpts) error
	Unfollow(ctx context.Context, fromUserID string, targetUserID string, opts *twitchhttp.ReqOpts) error
	HideAllFollows(ctx context.Context, userID string, opts *twitchhttp.ReqOpts) error
	RestoreAllFollows(ctx context.Context, userID string, opts *twitchhttp.ReqOpts) error
}

type client struct {
	twitchhttp.Client
}

// NewClient constructs a new client to consume the following service
func NewClient(conf twitchhttp.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchhttp.NewClient(conf)
	return &client{twitchClient}, err
}

// FollowResponse is the response from the follow endpoint
type followResponse struct {
	Follow *Follow `json:"follow"`
}

// Follow is an instance of a follow relationship between FromUserID and TargetUserID
type Follow struct {
	FollowedAt         time.Time `json:"followed_at"`
	FromUserID         string    `json:"from_user_id"`
	TargetUserID       string    `json:"target_user_id"`
	BlockNotifications bool      `json:"block_notifications"`
}

// Follows is the response received from the list follows endpoint
type Follows struct {
	Follows []*Follow `json:"follows"`
	Cursor  string    `json:"_cursor"`
}

// Followers is the response received from the list followers endpoint
type Followers struct {
	Followers []*Follow `json:"followers"`
	Cursor    string    `json:"_cursor"`
}

// FollowerCount is a count of followers for a user
type FollowerCount struct {
	Count int `json:"count"`
}

func (c *client) GetFollow(ctx context.Context, fromUserID string, targetUserID string, opts *twitchhttp.ReqOpts) (*Follow, error) {
	if fromUserID == targetUserID {
		return nil, &twitchhttp.Error{StatusCode: http.StatusNotFound, Message: fmt.Sprintf("A Follow between %s and %s was not found", fromUserID, targetUserID)}
	}

	path := fmt.Sprintf("v1/follows/%s/users/%s", fromUserID, targetUserID)

	result := &followResponse{}
	err := c.get(ctx, result, path, nil, opts, "service.follows.get_follow")
	return result.Follow, err
}

// ListFollowsParams contains all of the valid query parameters for the list follows endpoint
type ListFollowsParams struct {
	Cursor string
	Limit  int
	// SortDirection can be asc or desc, and defaults to asc
	SortDirection string
}

func (c *client) ListFollows(ctx context.Context, fromUserID string, params *ListFollowsParams, opts *twitchhttp.ReqOpts) (*Follows, error) {
	path := fmt.Sprintf("v1/follows/%s/users", fromUserID)
	query := url.Values{}
	if params != nil {
		if params.Cursor != "" {
			query.Add("cursor", params.Cursor)
		}
		if params.Limit != 0 {
			query.Add("limit", strconv.Itoa(params.Limit))
		}
		if params.SortDirection != "" {
			query.Add("direction", params.SortDirection)
		}
	}

	result := &Follows{}
	err := c.get(ctx, result, path, query, opts, "service.follows.list_follows")
	return result, err
}

// ListFollowersParams contains all of the valid parameters for the list followers endpoint
type ListFollowersParams struct {
	Cursor string
	Limit  int
	Offset int
	// SortDirection can be asc or desc, and defaults to asc
	SortDirection string
}

func (c *client) ListFollowers(ctx context.Context, fromUserID string, params *ListFollowersParams, opts *twitchhttp.ReqOpts) (*Followers, error) {
	path := fmt.Sprintf("v1/followers/%s/users", fromUserID)
	query := url.Values{}
	if params != nil {
		if params.Cursor != "" {
			query.Add("cursor", params.Cursor)
		}
		if params.Limit != 0 {
			query.Add("limit", strconv.Itoa(params.Limit))
		}
		if params.Offset != 0 {
			query.Add("offset", strconv.Itoa(params.Offset))
		}
		if params.SortDirection != "" {
			query.Add("direction", params.SortDirection)
		}
	}

	result := &Followers{}
	err := c.get(ctx, result, path, query, opts, "service.follows.list_followers")
	return result, err
}

func (c *client) BulkCountFollowers(ctx context.Context, userIDs []string, opts *twitchhttp.ReqOpts) (map[string]int, error) {
	followersCountMap := make(map[string]int, len(userIDs))

	var err error
	var mutex sync.Mutex
	var wg sync.WaitGroup

	wg.Add(len(userIDs))
	for _, userID := range userIDs {
		go func(userID string) {
			defer wg.Done()
			followersCount, cErr := c.CountFollowers(ctx, userID, opts)
			mutex.Lock()
			defer mutex.Unlock()
			followersCountMap[userID] = followersCount
			if cErr != nil {
				err = cErr
			}
		}(userID)
	}

	wg.Wait()
	return followersCountMap, err
}

func (c *client) CountFollowers(ctx context.Context, userID string, opts *twitchhttp.ReqOpts) (int, error) {
	path := fmt.Sprintf("v1/followers/%s/users/count", userID)
	result := &FollowerCount{}
	err := c.get(ctx, result, path, nil, opts, "service.follows.count_followers")
	return result.Count, err
}

func (c *client) HideAllFollows(ctx context.Context, userID string, opts *twitchhttp.ReqOpts) error {
	path := fmt.Sprintf("v1/follows/%s", userID)
	return c.delete(ctx, nil, path, nil, opts, "service.follows.hide_all_followers")
}

func (c *client) RestoreAllFollows(ctx context.Context, userID string, opts *twitchhttp.ReqOpts) error {
	path := fmt.Sprintf("v1/follows/%s/restore", userID)
	return c.post(ctx, nil, path, nil, nil, opts, "service.follows.restore_all_followers")
}

// FollowRequestBody contains configuration parameters for a follow
type FollowRequestBody struct {
	BlockNotifications bool `json:"block_notifications"`
}

func (c *client) Follow(ctx context.Context, fromUserID string, targetUserID string, body *FollowRequestBody, opts *twitchhttp.ReqOpts) error {
	path := fmt.Sprintf("v1/follows/%s/users/%s", fromUserID, targetUserID)
	return c.put(ctx, nil, path, nil, body, opts, "service.follows.upsert_follow")
}

func (c *client) Unfollow(ctx context.Context, fromUserID string, targetUserID string, opts *twitchhttp.ReqOpts) error {
	path := fmt.Sprintf("v1/follows/%s/users/%s", fromUserID, targetUserID)
	return c.delete(ctx, nil, path, nil, opts, "service.follows.unfollow")
}

func (c *client) get(
	ctx context.Context,
	into interface{},
	path string,
	queryParams url.Values,
	reqOpts *twitchhttp.ReqOpts,
	statName string) error {
	return c.http(ctx, into, "GET", path, queryParams, nil, reqOpts, statName)
}

func (c *client) delete(
	ctx context.Context,
	into interface{},
	path string,
	queryParams url.Values,
	reqOpts *twitchhttp.ReqOpts,
	statName string) error {
	return c.http(ctx, into, "DELETE", path, queryParams, nil, reqOpts, statName)
}

func (c *client) post(
	ctx context.Context,
	into interface{},
	path string,
	queryParams url.Values,
	body interface{},
	reqOpts *twitchhttp.ReqOpts,
	statName string) error {
	return c.http(ctx, into, "POST", path, queryParams, body, reqOpts, statName)
}

func (c *client) put(
	ctx context.Context,
	into interface{},
	path string,
	queryParams url.Values,
	body interface{},
	reqOpts *twitchhttp.ReqOpts,
	statName string) error {
	return c.http(ctx, into, "PUT", path, queryParams, body, reqOpts, statName)
}

func (c *client) http(
	ctx context.Context,
	into interface{},
	method string,
	path string,
	queryParams url.Values,
	body interface{},
	reqOpts *twitchhttp.ReqOpts,
	statName string) error {

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       statName,
		StatSampleRate: defaultStatSampleRate,
	})
	var bodyReader io.Reader
	if body != nil {
		httpBody, err := json.Marshal(body)
		if err != nil {
			return err
		}
		bodyReader = bytes.NewBuffer(httpBody)
	}
	req, err := c.NewRequest(method, path, bodyReader)
	if err != nil {
		return err
	}
	if queryParams != nil {
		req.URL.RawQuery = queryParams.Encode()
	}

	_, err = c.DoJSON(ctx, into, req, combinedReqOpts)
	return err
}
