package vinyl

import (
	"bytes"
	"encoding/json"
	"fmt"
	"time"
)

// Vod is a client-friendly representation of a VOD
type Vod struct {
	BroadcastID         int                     `json:"broadcast_id"`
	BroadcastType       string                  `json:"broadcast_type"`
	BroadcasterSoftware *string                 `json:"broadcaster_software"`
	CreatedAt           time.Time               `json:"created_at"`
	CreatedBy           *int64                  `json:"created_by_id"`
	DeleteAt            *time.Time              `json:"delete_at"`
	Deleted             *bool                   `json:"deleted"`
	Description         *string                 `json:"description"`
	DescriptionHTML     *string                 `json:"description_html"`
	Duration            int                     `json:"duration"`
	Fps                 map[string]float64      `json:"fps"`
	Game                *string                 `json:"game"`
	ID                  int64                   `json:"id"`
	Language            *string                 `json:"language"`
	Manifest            *string                 `json:"manifest"`
	Offset              int                     `json:"offset"`
	Origin              *string                 `json:"origin"`
	OwnerID             int64                   `json:"owner_id"`
	PublishedAt         *time.Time              `json:"published_at"`
	Resolutions         map[string]string       `json:"resolutions"`
	SourceArchiveID     *int64                  `json:"source_archive_id"`
	StartedOn           time.Time               `json:"recorded_on"`
	Status              string                  `json:"status"`
	TagList             string                  `json:"tag_list"`
	Title               *string                 `json:"title"`
	TotalLength         int                     `json:"total_length"`
	UpdatedAt           time.Time               `json:"updated_at"`
	URI                 string                  `json:"uri"`
	Viewable            *string                 `json:"viewable"`
	ViewableAt          *time.Time              `json:"viewable_at"`
	Views               int64                   `json:"views"`
	MutedSegments       []VodMutedSegment       `json:"muted_segments,omitempty"` // we actually would like to omit null instead of empty (not supported on json.Marshal) because we only want to include the field if loaded with "appeals_and_amrs" param, even if the returned list is empty. I'm not sure how to make an optional slice (slice or null) :feelsbadman:
	Notifications       map[string]Notification `json:"notifications,omitempty"`

	IncrementViewURL   string                            `json:"increment_view_count_url"`
	Qualities          []string                          `json:"qualities"`
	PreviewTemplate    string                            `json:"preview_template"`
	ShowFormats        map[string]map[string]interface{} `json:"show_formats"`
	ThumbnailTemplates ThumbnailTemplateList             `json:"thumbnail_templates"`
	SeekPreviewsURL    string                            `json:"seek_previews_url"`
	AnimatedPreviewURL string                            `json:"animated_preview_url"`
	Path               string                            `json:"path"`
	URL                string                            `json:"url"`

	// Fields associated with AMR & VOD Appeal
	AudibleMagicResponses []AMR         `json:"audible_magic_responses,omitempty"`
	VodAppeal             *VodAppeal    `json:"vod_appeal"`
	TrackAppeals          []TrackAppeal `json:"track_appeals"`
	IsMuted               *bool         `json:"is_muted"`
	CanCreateAppeal       *bool         `json:"can_create_appeal"`
	ViewOnlyAppeal        *bool         `json:"view_only_appeal"`
	HasPendingAppeal      *bool         `json:"has_pending_appeal"`

	// TODO: Deprecate this
	APIID string `json:"api_id"`
}

// ThumbnailTemplate contains information about a thumbnail template
type ThumbnailTemplate struct {
	Offset float64 `json:"offset,omitempty"`
	Path   string  `json:"path,omitempty"`
	Type   string  `json:"type,omitempty"`
	URL    string  `json:"url,omitempty"`
}

// ThumbnailTemplateList represents a list of ThumbnailTemplates
type ThumbnailTemplateList []ThumbnailTemplate

// VodMutedSegment one of the MutedSegments if included
type VodMutedSegment struct {
	Duration int `json:"duration"`
	Offset   int `json:"offset"`
}

// Notification is a notification setting for a vod
type Notification struct {
	ID         int64      `json:"id"`
	VodID      int64      `json:"vod_id"`
	Type       string     `json:"type"`
	CustomText string     `json:"custom_text"`
	Enabled    bool       `json:"enabled"`
	SentAt     *time.Time `json:"sent_at"`
	CreatedAt  time.Time  `json:"created_at"`
	UpdatedAt  time.Time  `json:"updated_at"`
}

// UnmarshalJSON unmarshals JSON into a ThumbnailTemplateList, accepting either a string or object.
func (t *ThumbnailTemplateList) UnmarshalJSON(data []byte) error {
	var i interface{}
	err := json.NewDecoder(bytes.NewReader(data)).Decode(&i)

	if err != nil {
		return err
	}

	switch i.(type) {
	case []interface{}:
		for _, template := range i.([]interface{}) {
			if template, ok := template.(map[string]interface{}); ok {
				newTemplate := ThumbnailTemplate{}

				data, err := json.Marshal(template)
				if err != nil {
					// ಠ_ಠ
					return err
				}

				err = json.Unmarshal(data, &newTemplate)
				if err != nil {
					return err
				}

				*t = append(*t, newTemplate)
			} else {
				return fmt.Errorf("unknown json type for ThumbnailTemplate: %T", template)
			}
		}
		return nil
	case string:
		return nil
	default:
		return fmt.Errorf("unknown json type for ThumbnailTemplateList: %T", i)
	}
}

// AMR is a client-friendly representation of an Audible Magic Response
type AMR struct {
	AudibleMagicResponseID int64      `json:"id"`
	VodID                  int64      `json:"vod_id"`
	Title                  *string    `json:"title"`
	Performer              *string    `json:"performer"`
	Genre                  *string    `json:"genre"`
	Artist                 *string    `json:"artist"`
	AlbumTitle             *string    `json:"album_title"`
	Song                   *string    `json:"song"`
	Isrc                   *string    `json:"isrc"`
	IsMatch                bool       `json:"is_match"`
	MatchOffsetSeconds     *int64     `json:"match_offset_seconds"`
	MatchDurationSeconds   *int64     `json:"match_duration_seconds"`
	ScanOffsetSeconds      *int64     `json:"scan_offset_seconds"`
	ScanDurationSeconds    *int64     `json:"scan_duration_seconds"`
	MuteOffsetSeconds      *int64     `json:"mute_offset_seconds"`
	MuteDurationSeconds    *int64     `json:"mute_duration_seconds"`
	CreatedAt              time.Time  `json:"created_at"`
	UpdatedAt              time.Time  `json:"updated_at"`
	SongLength             *int64     `json:"song_length_seconds"`
	UnmutedAt              *time.Time `json:"unmuted_at"`
	AmItemID               *string    `json:"am_item_id"`
}

// TrackAppeal is a client-friendly representation of a track appeal
type TrackAppeal struct {
	TrackAppealID          int64      `json:"id"`
	AudibleMagicResponseID int64      `json:"audible_magic_response_id"`
	VodAppealID            int64      `json:"vod_appeal_id"`
	Reason                 string     `json:"reason"`
	CreatedAt              time.Time  `json:"created_at"`
	UpdatedAt              time.Time  `json:"updated_at"`
	ResolvedAt             *time.Time `json:"resolved_at"`
}

// VodAppeal is a client-friendly representation of a VOD appeal
type VodAppeal struct {
	VodAppealID    int64      `json:"id"`
	ResolvedAt     *time.Time `json:"resolved_at"`
	CreatedAt      time.Time  `json:"created_at"`
	UpdatedAt      time.Time  `json:"updated_at"`
	VodID          int64      `json:"vod_id"`
	Priority       bool       `json:"priority"`
	FullName       string     `json:"full_name"`
	StreetAddress1 string     `json:"street_address_1"`
	StreetAddress2 string     `json:"street_address_2"`
	City           string     `json:"city"`
	State          string     `json:"state"`
	Zipcode        string     `json:"zipcode"`
	Country        string     `json:"country"`
}

// UserVODProperties is a client-friendly representation of the properties of vods belonging to a specific user.
type UserVODProperties struct {
	UserID                   int64     `json:"user_id"`
	SaveVODsForever          bool      `json:"save_vods_forever"`
	VODStorageDays           int       `json:"vod_storage_days"`
	CanUploadVOD             bool      `json:"can_upload_vod"`
	YoutubeExportingDisabled bool      `json:"youtube_exporting_disabled"`
	SkipUploadModeration     bool      `json:"skip_upload_moderation"`
	CreatedAt                time.Time `json:"created_at"`
	UpdatedAt                time.Time `json:"updated_at"`
}

// VodsAggregation is a client-friendly representation of a vods aggregation.
type VodsAggregation struct {
	TotalCount    uint64 `json:"total_count"`
	TotalDuration uint64 `json:"total_duration"`
}
