package dbtest

import (
	"database/sql"

	"code.justin.tv/chat/db"
	"github.com/DATA-DOG/go-sqlmock"
	"golang.org/x/net/context"
)

// MockDB is a mock implementation of db.DB.
// It runs on top of a regular sql.DB.
type MockDB struct {
	DB    *sql.DB
	dbCb  db.DBCallback
	runCb db.RunCallback
}

type mockRows struct {
	rows *sql.Rows
}

var _ db.Rows = new(mockRows)

func (r *mockRows) Close() error {
	return r.rows.Close()
}

func (r *mockRows) Next() bool {
	return r.rows.Next()
}

func (r *mockRows) Err() error {
	return r.rows.Err()
}

func (r *mockRows) Scan(dest ...interface{}) error {
	return r.rows.Scan(dest...)
}

func (r *mockRows) Columns() ([]string, error) {
	return r.rows.Columns()
}

type mockRow struct {
	row *sql.Row
}

func (r *mockRow) Scan(dest ...interface{}) error {
	return r.row.Scan(dest...)
}

type mockTx struct {
	tx *sql.Tx
}

func (tx *mockTx) Commit() error {
	return tx.tx.Commit()
}

func (tx *mockTx) Rollback() error {
	return tx.tx.Rollback()
}

func (tx *mockTx) Query(ctx context.Context, name, query string, args ...interface{}) (db.Rows, error) {
	rows, err := tx.tx.Query(query, args...)
	return &mockRows{rows: rows}, err
}

func (tx *mockTx) Exec(ctx context.Context, name, query string, args ...interface{}) (db.Result, error) {
	res, err := tx.tx.Exec(query, args...)
	return res, err
}

func (tx *mockTx) QueryRow(ctx context.Context, name string, query string, args ...interface{}) db.Row {
	row := tx.tx.QueryRow(query, args...)
	return &mockRow{row: row}
}

// NewMockDB creates a mock db.DB.
// If an underlying sql.DB is not provided,
// we create a mock sql.DB using sqlmock:
// http://github.com/DATA-DOG/go-sqlmock
func NewMockDB(db *sql.DB) db.DB {
	if db == nil {
		db, _, _ = sqlmock.New()
	}

	return &MockDB{DB: db}
}

// Begin returns an empty db.Tx struct.
func (d *MockDB) Begin(ctx context.Context, name string) (db.Tx, error) {
	tx, err := d.DB.Begin()
	return &mockTx{tx: tx}, err
}

// Open does nothing.
func (d *MockDB) Open() {
	// Do nothing
}

// Close does nothing.
func (d *MockDB) Close() error {
	return nil
}

// Reload does nothing.
func (d *MockDB) Reload(...db.Option) error {
	return nil
}

// Recycle does nothing.
func (d *MockDB) Recycle() error {
	return nil
}

// SetCallbacks sets event callbacks on the MockDB.
func (d *MockDB) SetCallbacks(dbCb db.DBCallback, runCb db.RunCallback) {
	d.dbCb = dbCb
	d.runCb = runCb
}

// Query calls Query on the underlying sql.DB, and wraps it in a db.Rows.
func (d *MockDB) Query(ctx context.Context, name, query string, args ...interface{}) (db.Rows, error) {
	rows, err := d.DB.Query(query, args...)
	return &mockRows{rows: rows}, err
}

// QueryRow calls QueryRow on the underlying sql.DB, and wraps it in a db.Row.
func (d *MockDB) QueryRow(ctx context.Context, name, query string, args ...interface{}) db.Row {
	row := d.DB.QueryRow(query, args...)
	return &mockRow{row: row}
}

// Exec calls Exec on the underlying sql.DB, and wraps it in a db.Result.
func (d *MockDB) Exec(ctx context.Context, name, query string, args ...interface{}) (db.Result, error) {
	res, err := d.DB.Exec(query, args...)
	return res, err
}

func (d *MockDB) Info() db.DBInfo {
	return db.DBInfo{MinAvailableConns: 1}
}
