package audrey

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
	"strconv"
	"time"

	"code.justin.tv/common/twitchhttp"
	"golang.org/x/net/context"
)

const (
	// NoComments explicitly requests no comments
	NoComments            = -1
	defaultStatSampleRate = 1.0
)

// Feed contains a single page of Posts with a Cursor that points to the next page
type Feed struct {
	Cursor string  `json:"cursor"`
	Topic  string  `json:"topic"`
	Posts  []*Post `json:"posts"`
}

// ChannelFeed is a Feed for a channel
type ChannelFeed struct {
	Feed
	ChannelID int `json:"channel_id"`
	Total     int `json:"total"`
}

type Emote struct {
	Id    int `json:"id"`
	Start int `json:"start"`
	End   int `json:"end"`
	Set   int `json:"set"`
}

// Post is the user facing representation of post
type Post struct {
	ID          string                     `json:"id"`
	CreatedAt   time.Time                  `json:"created_at"`
	Deleted     bool                       `json:"deleted"`
	UserID      int                        `json:"user_id"`
	Content     string                     `json:"body"`
	Permissions *PostPermissions           `json:"permissions,omitempty"`
	Emotes      []Emote                    `json:"emotes"`
	Embeds      []Embed                    `json:"embeds"`
	Reactions   map[string]*ReactionToItem `json:"reactions"`
	Comments    *Comments                  `json:"comments,omitempty"`
}

// PostPermissions contains a user's permissions to a post
type PostPermissions struct {
	CanReply    bool `json:"can_reply"`
	CanDelete   bool `json:"can_delete"`
	CanModerate bool `json:"can_moderate"`
}

// Comments is a paginated list of comments
type Comments struct {
	Total    int        `json:"total"`
	Cursor   string     `json:"cursor"`
	Comments []*Comment `json:"comments"`
}

// Comment is the user facing representation of a comment
type Comment struct {
	ID          string                     `json:"id"`
	CreatedAt   time.Time                  `json:"created_at"`
	Deleted     bool                       `json:"deleted"`
	UserID      int                        `json:"user_id"`
	Content     string                     `json:"body"`
	Emotes      []Emote                    `json:"emotes"`
	Permissions *CommentPermissions        `json:"permissions"`
	Reactions   map[string]*ReactionToItem `json:"reactions"`
}

// CommentPermissions contains a user's permissions to a comment
type CommentPermissions struct {
	CanDelete bool `json:"can_delete"`
}

// ReactionToItem contains info (counts, user IDs) about a single emote reaction to an item
type ReactionToItem struct {
	Emote   string `json:"emote"`
	Count   int    `json:"count"`
	UserIDs []int  `json:"user_ids"`
}

// Reaction is a reaction made by a user to a post or comment
type Reaction struct {
	ID        string    `json:"id"`
	CreatedAt time.Time `json:"created_at"`
	PostID    string    `json:"post_id"`
	CommentID string    `json:"comment_id"`
	UserID    int       `json:"user_id"`
	EmoteID   string    `json:"emote_id"`
}

// Embed contains VOD embed information
type Embed struct {
	Type         string    `json:"type,omitempty"`
	TwitchType   string    `json:"twitch_type,omitempty"`
	Title        string    `json:"title,omitempty"`
	Description  string    `json:"description,omitempty"`
	AuthorName   string    `json:"author_name,omitempty"`
	ThumbnailURL string    `json:"thumbnail_url,omitempty"`
	Game         string    `json:"game,omitempty"`
	PlayerHTML   string    `json:"player_html,omitempty"`
	CreatedAt    time.Time `json:"created_at,omitempty"`
	RequestURL   string    `json:"request_url"`
	VideoLength  int       `json:"video_length,omitempty"`
	ProviderName string    `json:"provider_name,omitempty"`
}

// UserSettings contains user setting information for a channel
type UserSettings struct {
	FriendsCanComment     *bool `json:"friends_can_comment,omitempty"`
	SubsCanComment        *bool `json:"subs_can_comment,omitempty"`
	UserDisabledComments  *bool `json:"user_disabled_comments,omitempty"`
	AdminDisabledComments *bool `json:"admin_disabled_comments,omitempty"`
}

// MaxBodySlurpSize is the number of bytes to try to copy from a response
var MaxBodySlurpSize = int64(2 << 10)

// Client is used by third parties to connect and interact with Audrey
type Client struct {
	Client         twitchhttp.Client
	StatSampleRate float32
}

type errInvalidStatusCode struct {
	Code int
}

func (e *errInvalidStatusCode) Error() string {
	return fmt.Sprintf("Invalid staus code: %d", e.Code)
}

func (c *Client) statSampleRate() float32 {
	if c.StatSampleRate == 0 {
		return defaultStatSampleRate
	}
	return c.StatSampleRate
}

func (c *Client) http(ctx context.Context, method string, statName string, path string, queryParams url.Values, body interface{}, reqOpts *twitchhttp.ReqOpts, into interface{}) error {
	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       statName,
		StatSampleRate: c.statSampleRate(),
	})

	var bodyReader io.Reader
	var bodyLength int
	if body != nil {
		httpBody, err := json.Marshal(body)
		if err != nil {
			return err
		}
		bodyReader = bytes.NewBuffer(httpBody)
		bodyLength = len(httpBody)
	}
	req, err := c.Client.NewRequest(method, path, bodyReader)
	if err != nil {
		return err
	}
	req.Cancel = ctx.Done()
	req.URL.RawQuery = queryParams.Encode()
	req.Header.Add("Content-Length", strconv.Itoa(bodyLength))

	resp, err := c.Client.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer func() {
		io.CopyN(ioutil.Discard, resp.Body, MaxBodySlurpSize)
		resp.Body.Close()
	}()
	if resp.StatusCode != http.StatusOK {
		return &errInvalidStatusCode{resp.StatusCode}
	}
	if err := json.NewDecoder(resp.Body).Decode(into); err != nil {
		return err
	}
	return nil
}

func (c *Client) get(ctx context.Context, statName string, path string, queryParams url.Values, reqOpts *twitchhttp.ReqOpts, into interface{}) error {
	return c.http(ctx, "GET", statName, path, queryParams, nil, reqOpts, into)
}

func (c *Client) delete(ctx context.Context, statName string, path string, queryParams url.Values, reqOpts *twitchhttp.ReqOpts, into interface{}) error {
	return c.http(ctx, "DELETE", statName, path, queryParams, nil, reqOpts, into)
}

func (c *Client) post(ctx context.Context, statName string, path string, queryParams url.Values, body interface{}, reqOpts *twitchhttp.ReqOpts, into interface{}) error {
	return c.http(ctx, "POST", statName, path, queryParams, body, reqOpts, into)
}

type RecentFeedParams struct {
	UserID string
	Limit  int
	Cursor string
}

// RecentFeed returns a time ordered recent feed from Audrey
func (c *Client) RecentFeed(ctx context.Context, params *RecentFeedParams, opts *twitchhttp.ReqOpts) (*Feed, error) {
	query := url.Values{}
	if params != nil {
		if params.UserID != "" {
			query.Add("user_id", params.UserID)
		}
		if params.Limit != 0 {
			query.Add("limit", strconv.Itoa(params.Limit))
		}
		if params.Cursor != "" {
			query.Add("last_key", params.Cursor)
		}
	}

	var ret Feed
	if err := c.get(ctx, "service.audrey.recent_feed", "v1/feed/recent", query, opts, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

// ChannelFeedParams defines params that can be used with Client.ChannelFeed()
type ChannelFeedParams struct {
	UserID   string
	Limit    int
	Cursor   string
	Comments int
}

// ChannelFeed returns the feed for the given channel.
func (c *Client) ChannelFeed(ctx context.Context, channelID string, params *ChannelFeedParams, opts *twitchhttp.ReqOpts) (*ChannelFeed, error) {
	path := fmt.Sprintf("v1/feed/%s/posts", channelID)
	query := url.Values{}
	if params != nil {
		if params.UserID != "" {
			query.Add("user_id", params.UserID)
		}
		if params.Limit != 0 {
			query.Add("limit", strconv.Itoa(params.Limit))
		}
		if params.Cursor != "" {
			query.Add("last_key", params.Cursor)
		}
		if params.Comments != 0 {
			if params.Comments == NoComments {
				query.Add("comments", "0")
			} else {
				query.Add("comments", strconv.Itoa(params.Comments))
			}
		}
	}
	var ret ChannelFeed
	if err := c.get(ctx, "service.audrey.get_feed", path, query, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// ChannelPermissions returns any permissions between channelID and userID
func (c *Client) ChannelPermissions(ctx context.Context, channelID string, userID string, opts *twitchhttp.ReqOpts) (*PostPermissions, error) {
	path := fmt.Sprintf("v1/feed/%s/permissions", channelID)
	query := url.Values{}
	query.Add("user_id", userID)

	var ret PostPermissions
	if err := c.get(ctx, "service.audrey.channel_permissions", path, query, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

type createPostBody struct {
	Content string `json:"content"`
}

// CreatePost create a post with the given content for the specified channel ID by the user ID
func (c *Client) CreatePost(ctx context.Context, channelID, userID string, content string, opts *twitchhttp.ReqOpts) (*Post, error) {
	path := fmt.Sprintf("v1/feed/%s/posts", channelID)
	query := url.Values{}
	query.Add("user_id", userID)
	body := createPostBody{Content: content}

	var ret Post
	if err := c.post(ctx, "service.audrey.create_post", path, query, body, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetPostParams defines params that can be used with Client.GetPost()
type GetPostParams struct {
	UserID   string
	Comments int
}

// GetPost gets the post for the given channel ID and post ID
func (c *Client) GetPost(ctx context.Context, channelID string, postID string, params *GetPostParams, opts *twitchhttp.ReqOpts) (*Post, error) {
	path := fmt.Sprintf("v1/feed/%s/posts/%s", channelID, postID)
	query := url.Values{}
	if params != nil {
		if params.UserID != "" {
			query.Add("user_id", params.UserID)
		}
		if params.Comments != 0 {
			if params.Comments == NoComments {
				query.Add("comments", "0")
			} else {
				query.Add("comments", strconv.Itoa(params.Comments))
			}
		}
	}

	var ret Post
	if err := c.get(ctx, "service.audrey.get_post", path, query, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// DeletePost deletes the post for the given ID as the user ID
func (c *Client) DeletePost(ctx context.Context, channelID, userID string, postID string, opts *twitchhttp.ReqOpts) (*Post, error) {
	path := fmt.Sprintf("v1/feed/%s/posts/%s", channelID, postID)
	query := url.Values{}
	query.Add("user_id", userID)

	var ret Post
	if err := c.delete(ctx, "service.audrey.delete_post", path, query, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

type createReactionBody struct {
	EmoteID string `json:"emote_id"`
}

// CreateReactionToPost reacts to the post for the given emote ID as the user ID
// Emote IDs are either a number specifying the emote, or "endorse"
func (c *Client) CreateReactionToPost(ctx context.Context, channelID, userID string, postID, emoteID string, opts *twitchhttp.ReqOpts) (*Reaction, error) {
	path := fmt.Sprintf("v1/feed/%s/posts/%s/reactions", channelID, postID)
	query := url.Values{}
	query.Add("user_id", userID)
	body := createReactionBody{EmoteID: emoteID}

	var ret Reaction
	if err := c.post(ctx, "service.audrey.create_reaction_to_post", path, query, body, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// DeleteReactionToPost deletes the reaction to the post for the given emote ID as the user ID
// Emote IDs are either a number specifying the emote, or "endorse"
func (c *Client) DeleteReactionToPost(ctx context.Context, channelID, userID string, postID, emoteID string, opts *twitchhttp.ReqOpts) (*Reaction, error) {
	path := fmt.Sprintf("v1/feed/%s/posts/%s/reactions/%s", channelID, postID, emoteID)
	query := url.Values{}
	query.Add("user_id", userID)

	var ret Reaction
	if err := c.delete(ctx, "service.audrey.delete_reaction_to_post", path, query, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetCommentsOfPostParams defines params that can be used with Client.GetCommentsOfPost()
type GetCommentsOfPostParams struct {
	UserID string
	Limit  int
	Cursor string
}

// GetCommentsOfPost gets the post for the given channel ID and post ID
func (c *Client) GetCommentsOfPost(ctx context.Context, channelID string, postID string, params *GetCommentsOfPostParams, opts *twitchhttp.ReqOpts) (*Comments, error) {
	path := fmt.Sprintf("v1/feed/%s/posts/%s/comments", channelID, postID)
	query := url.Values{}
	if params != nil {
		if params.UserID != "" {
			query.Add("user_id", params.UserID)
		}
		if params.Limit != 0 {
			query.Add("limit", strconv.Itoa(params.Limit))
		}
		if params.Cursor != "" {
			query.Add("last_key", params.Cursor)
		}
	}

	var ret Comments
	if err := c.get(ctx, "service.audrey.get_comments_of_post", path, query, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

type createCommentBody struct {
	Content string `json:"content"`
}

// CreateCommentToPost create a post with the given content for the specified channel ID by the user ID
func (c *Client) CreateCommentToPost(ctx context.Context, channelID, userID string, postID, content string, opts *twitchhttp.ReqOpts) (*Comment, error) {
	path := fmt.Sprintf("v1/feed/%s/posts/%s/comments", channelID, postID)
	query := url.Values{}
	query.Add("user_id", userID)
	body := createCommentBody{Content: content}

	var ret Comment
	if err := c.post(ctx, "service.audrey.create_comment_to_post", path, query, body, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// DeleteCommentsOfPostByAuthor deletes all of a user's comments for a post
func (c *Client) DeleteCommentsOfPostByAuthor(ctx context.Context, channelID, userID string, postID string, authorUserID string, opts *twitchhttp.ReqOpts) (*Comments, error) {
	path := fmt.Sprintf("v1/feed/%s/posts/%s/comments", channelID, postID)
	query := url.Values{}
	query.Add("user_id", userID)
	query.Add("comment_user_id", authorUserID)

	var ret Comments
	if err := c.delete(ctx, "service.audrey.delete_comments_of_post_by_author", path, query, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetCommentParams defines params that can be used with Client.GetComment()
type GetCommentParams struct {
	UserID int64
}

// GetComment gets the comment for the given channel ID and comment ID
func (c *Client) GetComment(ctx context.Context, channelID string, commentID string, params *GetCommentParams, opts *twitchhttp.ReqOpts) (*Comment, error) {
	path := fmt.Sprintf("v1/feed/%s/comments/%s", channelID, commentID)
	query := url.Values{}
	if params != nil {
		if params.UserID != 0 {
			query.Add("user_id", strconv.FormatInt(params.UserID, 10))
		}
	}

	var ret Comment
	if err := c.get(ctx, "service.audrey.get_comment", path, query, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// DeleteComment deletes the post for the given ID as the user ID
func (c *Client) DeleteComment(ctx context.Context, channelID, userID string, commentID string, opts *twitchhttp.ReqOpts) (*Comment, error) {
	path := fmt.Sprintf("v1/feed/%s/comments/%s", channelID, commentID)
	query := url.Values{}
	query.Add("user_id", userID)

	var ret Comment
	if err := c.delete(ctx, "service.audrey.delete_comment", path, query, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// CreateReactionToComment reacts to the comment for the given emote ID as the user ID
// Emote IDs are either a number specifying the emote, or "endorse"
func (c *Client) CreateReactionToComment(ctx context.Context, channelID, userID string, commentID, emoteID string, opts *twitchhttp.ReqOpts) (*Reaction, error) {
	path := fmt.Sprintf("v1/feed/%s/comments/%s/reactions", channelID, commentID)
	query := url.Values{}
	query.Add("user_id", userID)
	body := createReactionBody{EmoteID: emoteID}

	var ret Reaction
	if err := c.post(ctx, "service.audrey.create_reaction_to_comment", path, query, body, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// DeleteReactionToComment deletes the reaction to the comment for the given emote ID as the user ID
// Emote IDs are either a number specifying the emote, or "endorse"
func (c *Client) DeleteReactionToComment(ctx context.Context, channelID, userID string, commentID, emoteID string, opts *twitchhttp.ReqOpts) (*Reaction, error) {
	path := fmt.Sprintf("v1/feed/%s/comments/%s/reactions/%s", channelID, commentID, emoteID)
	query := url.Values{}
	query.Add("user_id", userID)

	var ret Reaction
	if err := c.delete(ctx, "service.audrey.delete_reaction_to_comment", path, query, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetSettings gets a user's settings for a channel
func (c *Client) GetSettings(ctx context.Context, channelID, userID string, opts *twitchhttp.ReqOpts) (*UserSettings, error) {
	path := fmt.Sprintf("v1/settings/%s", channelID)
	query := url.Values{}
	query.Add("user_id", userID)

	var ret UserSettings
	if err := c.get(ctx, "service.audrey.get_settings", path, query, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// UpdateSettings update's a user's settings for a channel
func (c *Client) UpdateSettings(ctx context.Context, channelID, userID string, settings *UserSettings, opts *twitchhttp.ReqOpts) (*UserSettings, error) {
	path := fmt.Sprintf("v1/settings/%s", channelID)
	query := url.Values{}
	query.Add("user_id", userID)

	var ret UserSettings
	if err := c.post(ctx, "service.audrey.create_post", path, query, settings, opts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}
