package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"

	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performGetAMRsForVodRequest(router *Server, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("GET", "http://localhost/v1/amrs", nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)

	return w
}

func TestGetAMRsForVod(t *testing.T) {
	Convey("GetAMRsForVod", t, func() {
		Convey("should fail with no vod_id", func() {
			router := mockServer()
			p := url.Values{}
			w := performGetAMRsForVodRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
		})

		Convey("should fail with empty vod_id", func() {
			router := mockServer()
			p := url.Values{"vod_id": []string{""}}
			w := performGetAMRsForVodRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
		})

		Convey("should fail with bogus vod_id", func() {
			router := mockServer()
			p := url.Values{"vod_id": []string{"asdfasdf"}}
			w := performGetAMRsForVodRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
		})

		Convey("should fail with backend error", func() {
			vodID := 12349
			router := mockServer()
			mockBackend.On("GetAMRsForVod", mock.Anything, vodID).Return(nil, fmt.Errorf("ads"))
			p := url.Values{"vod_id": []string{strconv.Itoa(vodID)}}
			w := performGetAMRsForVodRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("should succeed with an int vod_id", func() {
			vodID := 12349
			router := mockServer()
			mockBackend.On("GetAMRsForVod", mock.Anything, vodID).Return([]*models.AMR{}, nil)
			p := url.Values{"vod_id": []string{strconv.Itoa(vodID)}}
			w := performGetAMRsForVodRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusOK)
		})
	})
}
