package api

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/vod/vinyl/auth"
	"github.com/cactus/go-statsd-client/statsd"
	goji "goji.io"
	"goji.io/pat"
	"golang.org/x/net/context"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/vod/vinyl/backend"
	"code.justin.tv/vod/vinyl/errors"

	"code.justin.tv/common/golibs/errorlogger"
)

// Backend is a wrapper around Backender for mocking
type Backend interface {
	backend.Backender
}

// Server is the basic struct which contains the pieces needed to serve requests.
type Server struct {
	*goji.Mux
	Backend     Backend
	Stats       statsd.Statter
	ErrorLogger errorlogger.ErrorLogger
	authHandler auth.Handler
}

// NewServer generates a new server to handle API requests.
func NewServer(stats statsd.Statter, logger errorlogger.ErrorLogger, backend Backend, authHandler auth.Handler) (*Server, error) {
	server := twitchhttp.NewServer()

	s := &Server{
		Mux:         server,
		Backend:     backend,
		Stats:       stats,
		ErrorLogger: logger,
		authHandler: authHandler,
	}

	s.HandleFuncC(pat.Get("/v1/vods/related"), s.createHandler(s.Related, "one_off.related", 0.1))
	s.HandleFuncC(pat.Get("/v1/vods/top"), s.createHandler(s.Top, "one_off.top", 0.1))
	s.HandleFuncC(pat.Post("/v1/vods/followed"), s.createHandler(s.Followed, "one_off.followed", 0.1))
	s.HandleFuncC(pat.Put("/v1/vods/set_viewcounts"), s.createHandler(s.SetViewcounts, "one_off.set_viewcounts", 0.1))
	s.HandleFuncC(pat.Delete("/v1/vods/interval"), s.createHandler(s.DeleteVodsInterval, "one_off.delete_vods_interval", 0.1))

	s.HandleFuncC(pat.Get("/v1/vods/:vod_id/owner_for_rechat"), s.createHandler(s.OwnerForRechat, "vods.owner_for_rechat", 0.1))
	s.HandleFuncC(pat.Get("/v1/vods/:vod_id"), s.createHandler(s.GetVodByID, "vods.get_vod_by_id", 0.1))
	s.HandleFuncC(pat.Put("/v1/vods/:vod_id"), s.createHandler(s.UpdateVod, "vods.update", 0.1))
	s.HandleFuncC(pat.Post("/v1/vods/:vod_id/youtube_export"), s.createHandler(s.YoutubeExport, "vods.youtube_export", 1.0))
	s.HandleFuncC(pat.Get("/v1/vods"), s.createHandler(s.GetVodsByID, "vods.get_vods_by_id", 0.1))
	s.HandleFuncC(pat.Get("/v1/vods_aggregation"), s.createHandler(s.GetVodsAggregationByID, "vods.get_vods_aggregation_by_id", 0.1))
	s.HandleFuncC(pat.Post("/v1/vods"), s.createHandler(s.CreateVod, "vods.create_vod", 0.1))
	s.HandleFuncC(pat.Get("/v1/vods/user/:user_id"), s.createHandler(s.GetVodsByUser, "vods.get_vods_by_user", 0.1))
	s.HandleFuncC(pat.Get("/v1/vods/status/:status"), s.createHandler(s.GetVodsByStatus, "vods.get_vods_by_status", 0.1))
	s.HandleFuncC(pat.Post("/v1/vods/past_broadcast"), s.createHandler(s.CreatePastBroadcast, "vods.create_past_broadcast", 0.1))
	s.HandleFuncC(pat.Post("/v1/vods/highlight"), s.createHandler(s.CreateHighlight, "vods.create_highlight", 0.1))
	s.HandleFuncC(pat.Delete("/v1/vods/external"), s.createHandler(s.DeleteVodsExternal, "vods.delete_vods_external", 0.1))
	s.HandleFuncC(pat.Delete("/v1/vods"), s.createHandler(s.DeleteVods, "vods.delete_vods", 0.1))
	s.HandleFuncC(pat.Delete("/v1/users/:user_id/vods"), s.createHandler(s.DeleteUserVods, "vods.delete_user_vods", 0.1))
	s.HandleFuncC(pat.Post("/v1/vods/publish"), s.createHandler(s.PublishScheduledVods, "vods.publish_scheduled_vods", 1.0))
	s.HandleFuncC(pat.Get("/v1/vods/moderation/:vod_id"), s.createHandler(s.Moderation, "vods.moderation", 0.1))
	s.HandleFuncC(pat.Post("/v1/vods/accept"), s.createHandler(s.AcceptUnmoderatedVods, "vods.accept_unmoderated_vods", 1.0))
	s.HandleFuncC(pat.Post("/v1/vods/rebroadcast"), s.createHandler(s.Rebroadcast, "vods.rebroadcast", 1.0))

	s.HandleFuncC(pat.Get("/v1/amrs"), s.createHandler(s.GetAMRsForVod, "amrs.get_amrs_for_vod", 0.1))
	s.HandleFuncC(pat.Post("/v1/amrs"), s.createHandler(s.CreateAudibleMagicResponses, "amrs.create_amr", 0.1))
	s.HandleFuncC(pat.Put("/v1/amrs/:amr_id"), s.createHandler(s.UpdateAudibleMagicResponses, "amrs.update_amr", 0.1))

	s.HandleFuncC(pat.Get("/v1/vod_appeals"), s.createHandler(s.GetVodAppeals, "appeals.get_vod_appeals", 0.1))
	s.HandleFuncC(pat.Post("/v1/appeals"), s.createHandler(s.CreateAppeals, "appeals.create", 1.0))
	s.HandleFuncC(pat.Post("/v1/track_appeal/:appeal_id"), s.createHandler(s.ResolveTrackAppeal, "appeals.resolve_track_appeal", 0.1))
	s.HandleFuncC(pat.Post("/v1/vod_appeal/:appeal_id"), s.createHandler(s.ResolveVodAppeal, "appeals.resolve_vod_appeal", 0.1))

	s.HandleFuncC(pat.Get("/v1/user_video_privacy_properties/:user_id"), s.createHandler(s.GetUserVideoPrivacyProperties, "user_video_privacy_properties.get", 0.1))
	s.HandleFuncC(pat.Put("/v1/user_video_privacy_properties/:user_id"), s.createHandler(s.SetUserVideoPrivacyProperties, "user_video_privacy_properties.set", 0.1))
	s.HandleFuncC(pat.Get("/v1/user_vod_properties/:user_id"), s.createHandler(s.GetUserVODProperties, "user_vod_properties.get", 0.1))
	s.HandleFuncC(pat.Put("/v1/user_vod_properties/:user_id"), s.createHandler(s.SetUserVODProperties, "user_vod_properties.set", 0.1))

	s.HandleFuncC(pat.Post("/v1/vods/:vod_id/thumbnails"), s.createHandler(s.CreateThumbnails, "thumbnails.create", 0.1))
	s.HandleFuncC(pat.Delete("/v1/vods/:vod_id/thumbnails"), s.createHandler(s.DeleteThumbnail, "thumbnails.delete", 0.1))
	s.HandleFuncC(pat.Delete("/v1/vods/:vod_id/thumbnails/generated"), s.createHandler(s.DeleteThumbnails, "thumbnails.delete_generated", 0.1))

	return s, nil
}

// createHandler logs the request information. Includes timing but not response status.
func (s *Server) createHandler(handler func(context.Context, http.ResponseWriter, *http.Request), statName string, statRate float32) func(context.Context, http.ResponseWriter, *http.Request) {
	return func(ctx context.Context, w http.ResponseWriter, r *http.Request) {
		start := time.Now()
		endpointCtx := context.WithValue(ctx, "endpoint", statName)
		handler(endpointCtx, w, r)
		duration := time.Since(start)
		err := s.Stats.Inc(fmt.Sprintf("endpoints.%s", statName), 1, statRate)
		err = s.Stats.TimingDuration(fmt.Sprintf("endpoints.%s", statName), duration, statRate)
		if err != nil {
		}

		logString := fmt.Sprintf("%s - [%s] \"%s %s %s\" (%dms)",
			strings.Split(r.RemoteAddr, ":")[0],
			start.Format("02/Jan/2006:15:04:05 -0700"),
			r.Method,
			r.URL.RequestURI(),
			r.Proto,
			duration/time.Millisecond,
		)
		log.Printf(logString)
	}
}

func (s *Server) serveError(ctx context.Context, w http.ResponseWriter, r *http.Request, err error) {
	statusCode := errors.MatchError(err)

	var errorResp = map[string]interface{}{
		"status":  statusCode,
		"message": err.Error(),
		"error":   err.Error(), // TODO: once we've migrated to visage, delete this and make sure all clients are checking "message" field
	}

	w.WriteHeader(statusCode)

	endpoint := ctx.Value("endpoint").(string)
	if statusCode >= 400 && statusCode < 500 {
		logErr := s.Stats.Inc(fmt.Sprintf("endpoints.%s.error.%d", endpoint, statusCode), 1, 1)
		if logErr != nil {
		}
	} else {
		logErr := s.Stats.Inc(fmt.Sprintf("endpoints.%s.error", endpoint), 1, 1)
		if logErr != nil {
		}
		if s.ErrorLogger != nil {
			s.ErrorLogger.RequestError(r, err)
		}
	}

	s.serveJSON(w, r, errorResp)

}

func (s *Server) serveJSON(w http.ResponseWriter, r *http.Request, v interface{}) {
	content, err := json.MarshalIndent(v, "", "  ")
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	w.Header().Set("Content-Length", strconv.Itoa(len(content)))
	w.Header().Set("Content-Type", "application/json")
	_, err = w.Write(content)
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}
}
