package api

import (
	"encoding/json"
	"io/ioutil"
	"net/http"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"

	"golang.org/x/net/context"
)

// CreateAudibleMagicResponses supports creating audible magic response entries in the database.
func (s *Server) CreateAudibleMagicResponses(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	bodyBytes, err := ioutil.ReadAll(r.Body)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Could not read body: " + err.Error()})
		return
	}

	defer func() {
		closeErr := r.Body.Close()
		if closeErr != nil {
		}
	}()

	requiredFields := map[string][]string{}
	requiredFields["audible_magic_responses"] = []string{"vod_id", "is_match"}

	err = RequireFieldsForAllModels(bodyBytes, requiredFields)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	amrMap := map[string][]*models.AMR{}
	err = json.Unmarshal(bodyBytes, &amrMap)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Invalid body: " + err.Error()})
		return
	}

	amrs := amrMap["audible_magic_responses"]

	if len(amrs) < 1 {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Invalid body: nothing to create."})
		return

	}

	// This is a temporary restriction we will enforce for the migration.
	// Restricting the kensis write to a single VodID (shard) makes things simpler.
	// TODO: Remove after migration is complete.
	vodID := amrs[0].VodID
	for _, amr := range amrs {
		if amr.VodID != vodID {
			s.serveError(ctx, w, r,
				errors.RequestBodyError{ErrorString: "Invalid body: all audible magic responses must contain the same vod ID."})
			return
		}
	}

	createdAMRs, err := s.Backend.CreateAMRs(ctx, vodID, amrs)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	ret := map[string][]*models.AMR{
		"audible_magic_responses": createdAMRs,
	}

	s.serveJSON(w, r, ret)
}
