package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performCreateAMRRequest(router *Server, requestBody interface{}) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	body, err := json.Marshal(requestBody)
	So(err, ShouldBeNil)
	r, err := http.NewRequest("POST", "http://localhost/v1/amrs", bytes.NewBuffer(body))
	So(err, ShouldBeNil)

	router.ServeHTTP(w, r)

	return w
}

func TestCreateAMRs(t *testing.T) {
	Convey("Create AMRs", t, func() {
		Convey("should fail with a bogus body", func() {
			bogusBody := "thecowran"
			router := mockServer()
			w := performCreateAMRRequest(router, bogusBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid")
		})

		Convey("should fail with invalid amr parameters", func() {
			router := mockServer()
			amrMaps := []map[string]interface{}{}
			testMap := map[string]interface{}{
				"vod_id":              23,
				"is_match":            true,
				"mute_offset_seconds": "thisoffsetkeyshouldbeaninteger",
			}
			amrMaps = append(amrMaps, testMap)
			invalidBody := map[string][]map[string]interface{}{
				"audible_magic_responses": amrMaps,
			}

			w := performCreateAMRRequest(router, invalidBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Unrecognized type for integer")
		})

		Convey("should fail without the correct key in the body", func() {
			router := mockServer()
			amrMaps := []map[string]interface{}{}
			testMap := map[string]interface{}{
				"is_match":            true,
				"mute_offset_seconds": 23498,
			}
			amrMaps = append(amrMaps, testMap)
			invalidBody := map[string][]map[string]interface{}{
				"wrong key": amrMaps,
			}

			w := performCreateAMRRequest(router, invalidBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: audible_magic_responses")
		})

		Convey("should fail without the correct key in each of the individual maps", func() {
			router := mockServer()
			amrMaps := []map[string]interface{}{}
			testMap := map[string]interface{}{
				"vod_id":              123,
				"mute_offset_seconds": 23498,
			}
			amrMaps = append(amrMaps, testMap)
			invalidBody := map[string][]map[string]interface{}{
				"audible_magic_responses": amrMaps,
			}

			w := performCreateAMRRequest(router, invalidBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: is_match")
		})

		Convey("should fail with differing VOD ID's in the list of elements.", func() {
			router := mockServer()
			amrMaps := []map[string]interface{}{}
			testMap1 := map[string]interface{}{
				"vod_id":              123,
				"is_match":            true,
				"mute_offset_seconds": 23498,
			}
			testMap2 := map[string]interface{}{
				"vod_id":              124,
				"is_match":            true,
				"mute_offset_seconds": 23498,
			}
			amrMaps = append(amrMaps, testMap1, testMap2)
			invalidBody := map[string][]map[string]interface{}{
				"audible_magic_responses": amrMaps,
			}

			w := performCreateAMRRequest(router, invalidBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "all audible magic responses must contain the same vod ID")
		})

		Convey("should fail when no amrs", func() {
			router := mockServer()
			vodID := int64(123)
			amrMaps := []map[string]interface{}{}

			mockBackend.On("CreateAMRs", mock.Anything, vodID, mock.Anything).Return([]*models.AMR{}, nil)
			body := map[string][]map[string]interface{}{
				"audible_magic_responses": amrMaps,
			}

			w := performCreateAMRRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
		})

		Convey("should fail with backend error", func() {
			router := mockServer()
			vodID := int64(123)
			amrMaps := []map[string]interface{}{}
			testMap := map[string]interface{}{
				"vod_id":              vodID,
				"is_match":            true,
				"mute_offset_seconds": 23498,
			}
			amrMaps = append(amrMaps, testMap)

			mockBackend.On("CreateAMRs", mock.Anything, vodID, mock.Anything).Return(nil, fmt.Errorf("sdf"))
			body := map[string][]map[string]interface{}{
				"audible_magic_responses": amrMaps,
			}

			w := performCreateAMRRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("should succeed with the correct key in the body", func() {
			router := mockServer()
			vodID := int64(123)
			amrMaps := []map[string]interface{}{}
			testMap := map[string]interface{}{
				"vod_id":              vodID,
				"is_match":            true,
				"mute_offset_seconds": 23498,
			}
			amrMaps = append(amrMaps, testMap)

			mockBackend.On("CreateAMRs", mock.Anything, vodID, mock.Anything).Return([]*models.AMR{}, nil)
			body := map[string][]map[string]interface{}{
				"audible_magic_responses": amrMaps,
			}

			w := performCreateAMRRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusOK)
			So(w.Body.String(), ShouldContainSubstring, "audible_magic_responses")
		})
	})
}
