package api

import (
	"encoding/json"
	"io/ioutil"
	"net/http"
	"strconv"

	"code.justin.tv/common/goauthorization"
	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"
	"golang.org/x/net/context"
)

// CreateAppeals creates both VOD and track appeals for a particular vod.
func (s *Server) CreateAppeals(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	bodyBytes, err := ioutil.ReadAll(r.Body)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Could not read body: " + err.Error()})
		return
	}

	defer func() {
		closeErr := r.Body.Close()
		if closeErr != nil {
		}
	}()

	var inputs AppealsInput
	err = json.Unmarshal(bodyBytes, &inputs)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Invalid body: " + err.Error()})
		return
	}

	if err = inputs.validate(); err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	inputs.VodAppeal.VodID, err = strconv.ParseInt(inputs.VodID, 10, 64)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	err = s.authHandler.AuthorizeJWT(r, goauthorization.CapabilityClaims{
		"manage_appeals": goauthorization.CapabilityClaim{
			"vod_id": inputs.VodID,
		},
	})
	if err != nil {
		s.serveError(ctx, w, r, errors.UnauthorizedError{})
		return
	}

	resp, err := s.Backend.CreateAppeals(ctx, inputs.VodAppeal, inputs.TrackAppeals)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	s.serveJSON(w, r, resp)
}

// AppealsInput is the input format associated with the CreateAppeals endpoint.
type AppealsInput struct {
	VodID        string                `json:"vod_id"`
	VodAppeal    *models.VodAppeal     `json:"vod_appeal_params"`
	TrackAppeals []*models.TrackAppeal `json:"track_appeals"`
}

func (a AppealsInput) validate() error {
	if a.VodAppeal == nil {
		return errors.MissingParamError{ParamName: "vod_appeal_params"}
	}

	if a.TrackAppeals == nil || len(a.TrackAppeals) == 0 {
		return errors.MissingParamError{ParamName: "track_appeals"}
	}

	for _, ta := range a.TrackAppeals {
		if ta.AudibleMagicResponseID == 0 {
			return errors.MissingParamError{ParamName: "audible_magic_response_id"}
		}
		if ta.Reason == "" {
			return errors.MissingParamError{ParamName: "reason"}
		}
	}
	return nil
}
