package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/vod/vinyl/models"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performCreateAppealRequest(router *Server, requestBody interface{}) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	body, err := json.Marshal(requestBody)
	So(err, ShouldBeNil)
	r, err := http.NewRequest("POST", "http://localhost/v1/appeals", bytes.NewBuffer(body))
	So(err, ShouldBeNil)

	router.ServeHTTP(w, r)

	return w
}

func TestCreateAppeals(t *testing.T) {
	var inputs AppealsInput
	Convey("Create Appeals", t, func() {
		inputs = AppealsInput{
			VodID: "123",
			VodAppeal: &models.VodAppeal{
				FullName:       "First",
				StreetAddress1: "address",
				StreetAddress2: "address2",
				City:           "San Francisco",
				Zipcode:        "12345",
				Country:        "usa",
			},
			TrackAppeals: []*models.TrackAppeal{
				{
					AudibleMagicResponseID: 123,
					Reason:                 "fair_use",
				},
				{
					AudibleMagicResponseID: 124,
					Reason:                 "permission_rights",
				},
			},
		}
		Convey("should succeed when valid", func() {
			router := mockServer()
			mockBackend.On("CreateAppeals",
				mock.Anything,
				mock.MatchedBy(func(a *models.VodAppeal) bool {
					return a.FullName == inputs.VodAppeal.FullName
				}),
				mock.MatchedBy(func(a []*models.TrackAppeal) bool {
					return len(a) == len(inputs.TrackAppeals) && a[0].AudibleMagicResponseID == inputs.TrackAppeals[0].AudibleMagicResponseID
				}),
			).Return(map[string]interface{}{
				"vod_appeal":    nil,
				"track_appeals": nil,
			}, nil)
			w := performCreateAppealRequest(router, inputs)
			So(w.Code, ShouldEqual, http.StatusOK)
			So(w.Body.String(), ShouldContainSubstring, "vod_appeal")
			So(w.Body.String(), ShouldContainSubstring, "track_appeals")
		})

		Convey("should fail when backend error", func() {
			router := mockServer()
			mockBackend.On("CreateAppeals",
				mock.Anything,
				mock.MatchedBy(func(a *models.VodAppeal) bool {
					return a.FullName == inputs.VodAppeal.FullName
				}),
				mock.MatchedBy(func(a []*models.TrackAppeal) bool {
					return len(a) == len(inputs.TrackAppeals) && a[0].AudibleMagicResponseID == inputs.TrackAppeals[0].AudibleMagicResponseID
				}),
			).Return(nil, fmt.Errorf("sdfsdf"))
			w := performCreateAppealRequest(router, inputs)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("should fail with a bogus body", func() {
			router := mockServer()
			bogusBody := "thecowran"
			w := performCreateAppealRequest(router, bogusBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid")
		})

		Convey("vod_appeal_params", func() {
			Convey("should fail when vod_appeal_params is nil", func() {
				router := mockServer()
				inputs.VodAppeal = nil
				w := performCreateAppealRequest(router, inputs)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "Missing required param: vod_appeal_params")
			})
		})

		Convey("track_appeals", func() {
			Convey("should fail when track_appeals is nil", func() {
				router := mockServer()
				inputs.TrackAppeals = nil
				w := performCreateAppealRequest(router, inputs)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "Missing required param: track_appeals")
			})

			Convey("should fail when track_appeals is empty", func() {
				router := mockServer()
				inputs.TrackAppeals = []*models.TrackAppeal{}
				w := performCreateAppealRequest(router, inputs)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "Missing required param: track_appeals")
			})

			Convey("should fail without the correct key in each of the individual maps", func() {
				router := mockServer()
				inputs.TrackAppeals = []*models.TrackAppeal{
					{AudibleMagicResponseID: 123},
					{AudibleMagicResponseID: 124},
				}

				w := performCreateAppealRequest(router, inputs)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "Missing required param: reason")
			})

		})
	})
}
