package api

import (
	"encoding/json"
	"io/ioutil"
	"net/http"
	"strconv"

	"code.justin.tv/common/goauthorization"
	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"

	"golang.org/x/net/context"
)

// CreateHighlight creates a database entry for a highlight in the corresponding table.
func (s *Server) CreateHighlight(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	bodyBytes, err := ioutil.ReadAll(r.Body)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Could not read body: " + err.Error()})
		return
	}

	defer func() {
		closeErr := r.Body.Close()
		if closeErr != nil {
		}
	}()

	highlightInfo := map[string]*models.HighlightInput{}
	err = json.Unmarshal(bodyBytes, &highlightInfo)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Invalid body: " + err.Error()})
		return
	}

	highlight, ok := highlightInfo["highlight"]
	if !ok {
		s.serveError(ctx, w, r, errors.MissingParamError{ParamName: "highlight"})
		return
	}

	if highlight.VodID == 0 {
		s.serveError(ctx, w, r, errors.MissingParamError{ParamName: "id"})
		return
	}

	err = s.authHandler.AuthorizeJWT(r, goauthorization.CapabilityClaims{
		"manage_vod": goauthorization.CapabilityClaim{
			"vod_id": strconv.FormatInt(highlight.VodID, 10),
		},
	})
	if err != nil {
		s.serveError(ctx, w, r, errors.UnauthorizedError{})
		return
	}

	err = validateDurationAndOffset(highlight)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	createdHighlight, err := s.Backend.CreateHighlight(ctx, r, highlight)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	ret := map[string]*models.Vod{
		"highlight": createdHighlight,
	}

	s.serveJSON(w, r, ret)

}
