package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performCreateHighlightRequest(router *Server, requestBody interface{}) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	body, err := json.Marshal(requestBody)
	So(err, ShouldBeNil)
	r, err := http.NewRequest("POST", "http://localhost/v1/vods/highlight", bytes.NewBuffer(body))
	So(err, ShouldBeNil)

	router.ServeHTTP(w, r)
	return w
}

func TestCreateHighlight(t *testing.T) {
	Convey("Creating a highlight", t, func() {
		Convey("should fail with a bogus body", func() {
			router := mockServer()
			bogusBody := "thecowran"
			w := performCreateHighlightRequest(router, bogusBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid")
		})

		Convey("should fail with invalid vod parameters", func() {
			router := mockServer()
			testMap := map[string]interface{}{
				"id": "should be integer",
			}
			invalidBody := map[string]map[string]interface{}{
				"highlight": testMap,
			}

			w := performCreateHighlightRequest(router, invalidBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "cannot unmarshal string")
		})

		Convey("should fail without the correct key in the body", func() {
			router := mockServer()
			testMap := map[string]interface{}{
				"id": 24,
			}
			invalidBody := map[string]map[string]interface{}{
				"wrong key": testMap,
			}

			w := performCreateHighlightRequest(router, invalidBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: highlight")
		})

		Convey("should fail for no vod id", func() {
			router := mockServer()
			testMap := map[string]interface{}{
				"id": 0,
			}
			body := map[string]map[string]interface{}{
				"highlight": testMap,
			}

			w := performCreateHighlightRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
		})

		Convey("should fail for bad times", func() {
			router := mockServer()
			testMap := map[string]interface{}{
				"id":         1,
				"start_time": int64(10),
				"end_time":   int64(0),
			}
			body := map[string]map[string]interface{}{
				"highlight": testMap,
			}

			w := performCreateHighlightRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
		})

		Convey("should fail for backend error", func() {
			router := mockServer()
			vodID := int64(123)
			testMap := map[string]interface{}{
				"id":         vodID,
				"start_time": int64(0),
				"end_time":   int64(10),
			}
			body := map[string]map[string]interface{}{
				"highlight": testMap,
			}
			mockBackend.On("CreateHighlight", mock.Anything, mock.Anything, mock.MatchedBy(func(a *models.HighlightInput) bool {
				return a.VodID == vodID
			})).Return(nil, fmt.Errorf("sdfdf"))

			w := performCreateHighlightRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("should succeed all the param checks with the correct key in the body", func() {
			router := mockServer()
			vodID := int64(123)
			testMap := map[string]interface{}{
				"id":         vodID,
				"start_time": int64(0),
				"end_time":   int64(10),
			}
			body := map[string]map[string]interface{}{
				"highlight": testMap,
			}
			mockBackend.On("CreateHighlight", mock.Anything, mock.Anything, mock.MatchedBy(func(a *models.HighlightInput) bool {
				return a.VodID == vodID
			})).Return(&models.Vod{Status: "whatever"}, nil)

			w := performCreateHighlightRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusOK)
			So(w.Body.String(), ShouldContainSubstring, "whatever")
		})
	})
}
