package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performCreatePastBroadcastRequest(router *Server, requestBody interface{}) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	body, err := json.Marshal(requestBody)
	So(err, ShouldBeNil)
	r, err := http.NewRequest("POST", "http://localhost/v1/vods/past_broadcast", bytes.NewBuffer(body))
	So(err, ShouldBeNil)

	router.ServeHTTP(w, r)
	return w
}

func TestCreatePastBroadcast(t *testing.T) {
	Convey("Create past broadcasts", t, func() {
		Convey("should fail with a bogus body", func() {
			router := mockServer()
			bogusBody := "thecowran"
			w := performCreatePastBroadcastRequest(router, bogusBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid")
		})

		Convey("should fail with invalid vod parameters", func() {
			router := mockServer()
			testMap := map[string]interface{}{
				"broadcast_id": "should be integer",
				"uri":          "unique uri",
			}
			invalidBody := map[string]map[string]interface{}{
				"past_broadcast": testMap,
			}

			w := performCreatePastBroadcastRequest(router, invalidBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "cannot unmarshal string")
		})

		Convey("should fail without the correct key in the body", func() {
			router := mockServer()
			testMap := map[string]interface{}{
				"broadcast_id": 24,
			}
			invalidBody := map[string]map[string]interface{}{
				"wrong key": testMap,
			}

			w := performCreatePastBroadcastRequest(router, invalidBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: past_broadcast")
		})

		Convey("should fail without the correct key in each of the individual maps", func() {
			router := mockServer()
			testMap := map[string]interface{}{
				"uri": "hello",
			}
			invalidBody := map[string]map[string]interface{}{
				"past_broadcast": testMap,
			}

			w := performCreatePastBroadcastRequest(router, invalidBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: broadcast_id")
		})

		Convey("should fail without url", func() {
			router := mockServer()
			broadcastID := 123
			testMap := map[string]interface{}{
				"broadcast_id": broadcastID,
				"uri":          "",
			}

			body := map[string]map[string]interface{}{
				"past_broadcast": testMap,
			}

			w := performCreatePastBroadcastRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
		})

		Convey("should fail with backend error", func() {
			router := mockServer()
			broadcastID := 123
			testMap := map[string]interface{}{
				"broadcast_id": broadcastID,
				"uri":          "unique uri",
			}
			mockBackend.On("CreatePastBroadcast", mock.Anything, mock.Anything, mock.MatchedBy(func(a *models.Vod) bool {
				return a.BroadcastID == broadcastID
			})).Return(nil, fmt.Errorf("sdfdf"))

			body := map[string]map[string]interface{}{
				"past_broadcast": testMap,
			}

			w := performCreatePastBroadcastRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("should succeed with the correct key in the body", func() {
			router := mockServer()
			broadcastID := 123
			testMap := map[string]interface{}{
				"broadcast_id": broadcastID,
				"uri":          "unique uri",
			}
			mockBackend.On("CreatePastBroadcast", mock.Anything, mock.Anything, mock.MatchedBy(func(a *models.Vod) bool {
				return a.BroadcastID == broadcastID
			})).Return(nil, nil)

			body := map[string]map[string]interface{}{
				"past_broadcast": testMap,
			}

			w := performCreatePastBroadcastRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusOK)
		})
	})
}
