package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func createThumbnailRequest(router *Server, requestBody interface{}) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	body, err := json.Marshal(requestBody)
	So(err, ShouldBeNil)
	r, err := http.NewRequest("POST", "http://localhost/v1/vods/1234/thumbnails", bytes.NewBuffer(body))
	So(err, ShouldBeNil)

	router.ServeHTTP(w, r)
	return w
}

func TestCreateThumbnail(t *testing.T) {
	Convey("CreateThumbnail", t, func() {
		Convey("should fail with an invalid body", func() {
			router := mockServer()
			invalidBody := "nonsense"
			w := createThumbnailRequest(router, invalidBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid")
		})

		Convey("should fail without the correct key in the body", func() {
			router := mockServer()
			testMap := map[string]interface{}{
				"path": "some_image.jpg",
			}
			invalidBody := map[string]map[string]interface{}{
				"wrong key": testMap,
			}

			w := createThumbnailRequest(router, invalidBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid body")
		})

		Convey("should fail without thumbnails field", func() {
			router := mockServer()

			body := map[string][]map[string]interface{}{}
			w := createThumbnailRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
		})

		Convey("should fail with backend error", func() {
			router := mockServer()
			thumbnail := map[string]interface{}{
				"path":   "some_image.jpg",
				"offset": 123.123,
			}
			mockBackend.On("CreateThumbnails", mock.Anything, int64(1234), mock.MatchedBy(func(a models.Thumbnails) bool {
				return len(a) == 1 && a[0].Path == thumbnail["path"]
			})).Return(nil, fmt.Errorf("sdfd"))
			body := map[string][]map[string]interface{}{
				"thumbnails": []map[string]interface{}{thumbnail},
			}
			w := createThumbnailRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("should succeed with valid input", func() {
			router := mockServer()
			thumbnail := map[string]interface{}{
				"path":   "some_image.jpg",
				"offset": 123.123,
			}
			mockBackend.On("CreateThumbnails", mock.Anything, int64(1234), mock.MatchedBy(func(a models.Thumbnails) bool {
				return len(a) == 1 && a[0].Path == thumbnail["path"]
			})).Return(nil, nil)
			body := map[string][]map[string]interface{}{
				"thumbnails": []map[string]interface{}{thumbnail},
			}
			w := createThumbnailRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusOK)
		})
	})
}
