package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performCreateVod(router *Server, requestBody interface{}) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	body, err := json.Marshal(requestBody)
	So(err, ShouldBeNil)
	r, err := http.NewRequest("POST", "http://localhost/v1/vods", bytes.NewBuffer(body))
	So(err, ShouldBeNil)

	router.ServeHTTP(w, r)
	return w
}

func TestCreateVod(t *testing.T) {
	Convey("Create past broadcasts", t, func() {
		Convey("should fail with a bogus body", func() {
			router := mockServer()
			bogusBody := "thecowran"
			w := performCreateVod(router, bogusBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid")
		})

		Convey("should fail without the correct key in the body", func() {
			router := mockServer()
			testMap := map[string]interface{}{
				"broadcast_id": 24,
			}
			invalidBody := map[string]map[string]interface{}{
				"wrong key": testMap,
			}

			w := performCreateVod(router, invalidBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: vod")
		})

		Convey("should fail with backend error", func() {
			router := mockServer()
			broadcastID := 123
			testMap := map[string]interface{}{
				"broadcast_id": broadcastID,
				"uri":          "unique uri",
			}
			mockBackend.On("CreateVod", mock.Anything, mock.Anything, mock.MatchedBy(func(a *models.Vod) bool {
				return a.BroadcastID == broadcastID
			})).Return(nil, fmt.Errorf("dfsdf"))
			body := map[string]map[string]interface{}{
				"vod": testMap,
			}

			w := performCreateVod(router, body)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("should succeed with the correct key in the body", func() {
			router := mockServer()
			broadcastID := 123
			testMap := map[string]interface{}{
				"broadcast_id": broadcastID,
				"uri":          "unique uri",
			}
			mockBackend.On("CreateVod", mock.Anything, mock.Anything, mock.MatchedBy(func(a *models.Vod) bool {
				return a.BroadcastID == broadcastID
			})).Return(&models.Vod{Status: "sdfd"}, nil)
			body := map[string]map[string]interface{}{
				"vod": testMap,
			}

			w := performCreateVod(router, body)
			So(w.Code, ShouldEqual, http.StatusOK)
			So(w.Body.String(), ShouldContainSubstring, "sdfd")
		})
	})
}
