package api

import (
	"net/http"
	"strconv"
	"strings"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"
	"golang.org/x/net/context"
)

// DeleteVods removes a set of VOD entries from the corresponding table.
func (s *Server) DeleteVods(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	queryValues := r.URL.Query()
	vodIDsParam := queryValues.Get("ids")
	if vodIDsParam == "" {
		s.serveError(ctx, w, r, errors.MissingParamError{ParamName: "ids"})
		return
	}
	vodIDs, err := getVodIDs(vodIDsParam)
	if err != nil {
		s.serveError(ctx, w, r, errors.IntegerParseError{ParamName: "ids", ParamValue: vodIDsParam})
		return
	}
	destructive := queryValues.Get("destructive") == "true"
	if destructive {
		err = s.Backend.DeleteVods(ctx, vodIDs)
		if err != nil {
			s.serveError(ctx, w, r, err)
			return
		}
	} else {
		for _, vodID := range vodIDs {
			_, err = s.Backend.UpdateVod(ctx, r, vodID, models.VodUpdateInput{
				Deleted: models.NullBool{Bool: true, Valid: true},
			})
			if err != nil {
				s.serveError(ctx, w, r, err)
				return
			}
		}
	}

	w.WriteHeader(http.StatusNoContent)
}

func getVodIDs(vodIDsParam string) ([]int64, error) {
	var vodIDs []int64
	vodIDParams := strings.Split(vodIDsParam, ",")
	for _, vodIDParam := range vodIDParams {
		vodID, err := strconv.Atoi(vodIDParam)
		if err != nil {
			return nil, err
		}
		vodIDs = append(vodIDs, int64(vodID))
	}
	return vodIDs, nil
}
