package api

import (
	"net/http"

	"code.justin.tv/common/goauthorization"
	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"
	"golang.org/x/net/context"
)

// DeleteVodsExternal removes a set of VOD entries from the corresponding table.
func (s *Server) DeleteVodsExternal(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	queryValues := r.URL.Query()
	vodIDsParam := queryValues.Get("ids")
	if vodIDsParam == "" {
		s.serveError(ctx, w, r, errors.MissingParamError{ParamName: "ids"})
		return
	}
	vodIDs, err := getVodIDs(vodIDsParam)
	if err != nil {
		s.serveError(ctx, w, r, errors.IntegerParseError{ParamName: "ids", ParamValue: vodIDsParam})
		return
	}

	err = s.authHandler.AuthorizeJWT(r, goauthorization.CapabilityClaims{
		"manage_vod": goauthorization.CapabilityClaim{
			"vod_id": vodIDsParam,
		},
	})
	if err != nil {
		s.serveError(ctx, w, r, errors.UnauthorizedError{})
		return
	}

	destructive := queryValues.Get("destructive") == "true"
	if destructive {
		err = s.Backend.DeleteVods(ctx, vodIDs)
		if err != nil {
			s.serveError(ctx, w, r, err)
			return
		}
	} else {
		for _, vodID := range vodIDs {
			_, err = s.Backend.UpdateVod(ctx, r, vodID, models.VodUpdateInput{
				Deleted: models.NullBool{Bool: true, Valid: true},
			})
			if err != nil {
				s.serveError(ctx, w, r, err)
				return
			}
		}
	}

	w.WriteHeader(http.StatusNoContent)
}
