package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	"code.justin.tv/common/config"
	"code.justin.tv/vod/vinyl/api/mocks"
	"code.justin.tv/vod/vinyl/auth"
	"code.justin.tv/vod/vinyl/models"

	"github.com/cactus/go-statsd-client/statsd"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performDeleteVodsExternalRequest(router *Server, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("DELETE", "http://0.0.0.0/v1/vods/external", nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)

	return w
}

func TestDeleteVodsExternal(t *testing.T) {
	Convey("DeleteVodsExternal", t, func() {
		Convey("Fails when vod_ids is missing", func() {
			router := mockServer()
			p := url.Values{}
			w := performDeleteVodsExternalRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: ids")
		})
		Convey("Fails when given an empty list of ids", func() {
			router := mockServer()
			p := url.Values{
				"ids": []string{""},
			}
			w := performDeleteVodsExternalRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: ids")
		})
		Convey("Fails when given invalid ids", func() {
			router := mockServer()
			p := url.Values{
				"ids": []string{"123,abc"},
			}
			w := performDeleteVodsExternalRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "ids is not a number:")
		})
		Convey("Succeeds when given a valid vod id", func() {
			router := mockServer()
			p := url.Values{
				"ids": []string{"1234"},
			}
			mockBackend.On("UpdateVod", mock.Anything, mock.Anything, int64(1234), models.VodUpdateInput{
				Deleted: models.NullBool{Bool: true, Valid: true},
			}).Return(nil, nil)
			w := performDeleteVodsExternalRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusNoContent)
		})
		Convey("fails when update fails", func() {
			router := mockServer()
			p := url.Values{
				"ids": []string{"1234"},
			}
			mockBackend.On("UpdateVod", mock.Anything, mock.Anything, int64(1234), models.VodUpdateInput{
				Deleted: models.NullBool{Bool: true, Valid: true},
			}).Return(nil, fmt.Errorf("sdfd"))
			w := performDeleteVodsExternalRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("Succeeds when given valid ids", func() {
			router := mockServer()
			p := url.Values{
				"ids": []string{"1,2"},
			}
			mockBackend.On("UpdateVod", mock.Anything, mock.Anything, int64(1), models.VodUpdateInput{
				Deleted: models.NullBool{Bool: true, Valid: true},
			}).Return(nil, nil)
			mockBackend.On("UpdateVod", mock.Anything, mock.Anything, int64(2), models.VodUpdateInput{
				Deleted: models.NullBool{Bool: true, Valid: true},
			}).Return(nil, nil)
			w := performDeleteVodsExternalRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusNoContent)
		})
		Convey("fails when delete fails", func() {
			router := mockServer()
			vodIDs := []int64{123, 456}
			mockBackend.On("DeleteVods", mock.Anything, vodIDs).Return(fmt.Errorf("adf"))
			p := url.Values{
				"ids":         []string{"123,456"},
				"destructive": []string{"true"},
			}
			w := performDeleteVodsExternalRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})
		Convey("Succeeds when destructive is included", func() {
			router := mockServer()
			vodIDs := []int64{123, 456}
			mockBackend.On("DeleteVods", mock.Anything, vodIDs).Return(nil)
			p := url.Values{
				"ids":         []string{"123,456"},
				"destructive": []string{"true"},
			}
			w := performDeleteVodsExternalRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusNoContent)
		})
		Convey("fails when auth fails", func() {
			// reconfigure the request object to fake an auth failure
			w := httptest.NewRecorder()
			stats, _ := statsd.NewNoopClient()
			logger := config.RollbarErrorLogger()
			vodIDs := []int64{123}
			mockBackend := &mocks.Backend{}
			mockBackend.On("DeleteVods", mock.Anything, vodIDs).Return(nil)
			authThatFails := auth.NewFakeHandler(false)
			server, _ := NewServer(stats, logger, mockBackend, authThatFails)

			r, err := http.NewRequest("DELETE", "http://localhost/v1/vods/external", nil)
			p := url.Values{
				"ids":         []string{"123"},
				"destructive": []string{"false"},
			}
			So(err, ShouldBeNil)
			r.URL.RawQuery = p.Encode()
			server.ServeHTTP(w, r)

			// Now check that the status is Unauthorized
			So(w.Code, ShouldEqual, http.StatusUnauthorized)
		})
	})
}
