package api

import (
	"net/http"
	"strconv"
	"time"

	"code.justin.tv/vod/vinyl/errors"
	"golang.org/x/net/context"
)

// DeleteVodsInterval takes a time interval and deletes all vods that need to be deleted in that interval
func (s *Server) DeleteVodsInterval(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	query := r.URL.Query()

	startTimeParam := query.Get("start_time")
	if startTimeParam == "" {
		s.serveError(ctx, w, r, errors.MissingParamError{ParamName: "start_time"})
		return
	}
	startTime, err := timeFromSecondsParam(startTimeParam)
	if err != nil {
		s.serveError(ctx, w, r, errors.TimeParseError{ParamName: "start_time", ParamValue: startTimeParam})
		return
	}

	endTimeParam := query.Get("end_time")
	if endTimeParam == "" {
		s.serveError(ctx, w, r, errors.MissingParamError{ParamName: "end_time"})
		return
	}
	endTime, err := timeFromSecondsParam(endTimeParam)
	if err != nil {
		s.serveError(ctx, w, r, errors.TimeParseError{ParamName: "end_time", ParamValue: endTimeParam})
		return
	}

	if startTime.After(endTime) {
		s.serveError(ctx, w, r, errors.InvalidTimeInterval{
			StartTimeName: "start_time", StartTimeValue: startTime,
			EndTimeName: "end_time", EndTimeValue: endTime})
		return
	}

	go func() {
		_, err := s.Backend.DeleteVodsInterval(context.Background(), startTime, endTime)
		if err != nil {
		}
	}()

	w.WriteHeader(http.StatusNoContent)
}

func timeFromSecondsParam(param string) (time.Time, error) {
	seconds, err := strconv.ParseInt(param, 10, 64)
	if err != nil {
		return time.Now(), err
	}
	return time.Unix(seconds, 0).UTC(), nil
}
