package api

import (
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"
	"time"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performDeleteVodsIntervalRequest(router *Server, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("DELETE", "http://0.0.0.0/v1/vods/interval", nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)

	return w
}

func TestDeleteVodsInterval(t *testing.T) {
	Convey("DeleteVodsInterval", t, func() {
		Convey("Fails when start_time is missing", func() {
			router := mockServer()
			p := url.Values{
				"end_time": []string{"Fri, 10 Jun 2016 21:21:03 +0000"},
			}
			w := performDeleteVodsIntervalRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: start_time")
		})
		Convey("Fails when end_time is missing", func() {
			router := mockServer()
			p := url.Values{
				"start_time": []string{"1466749430"},
			}
			w := performDeleteVodsIntervalRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: end_time")
		})
		Convey("Fails when start_time is not a valid time", func() {
			router := mockServer()
			p := url.Values{
				"start_time": []string{"146674943027301928730918273098172039871029837"},
				"end_time":   []string{"1466749439"},
			}
			w := performDeleteVodsIntervalRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "start_time is not a valid time")
		})
		Convey("Fails when end_time is not a valid time", func() {
			router := mockServer()
			p := url.Values{
				"start_time": []string{"1466749430"},
				"end_time":   []string{"Fri, 10 Jun 21:21:03 +0000"},
			}
			w := performDeleteVodsIntervalRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "end_time is not a valid time")
		})
		Convey("Fails when end_time is before start_time", func() {
			router := mockServer()
			p := url.Values{
				"start_time": []string{"1466749439"},
				"end_time":   []string{"1466749430"},
			}
			w := performDeleteVodsIntervalRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "start_time must be before end_time")
		})

		Convey("Succeeds when start_time and end_time are valid", func() {
			router := mockServer()
			start := int64(1466749430)
			end := int64(1466749439)
			mockBackend.On("DeleteVodsInterval", mock.Anything,
				mock.MatchedBy(func(t time.Time) bool {
					return t.UTC().Unix() == start
				}),
				mock.MatchedBy(func(t time.Time) bool {
					return t.UTC().Unix() == end
				}),
			).Return(0, nil)
			p := url.Values{
				"start_time": []string{strconv.FormatInt(start, 10)},
				"end_time":   []string{strconv.FormatInt(end, 10)},
			}
			w := performDeleteVodsIntervalRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusNoContent)
		})
	})
}
