package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performDeleteVodsRequest(router *Server, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("DELETE", "http://0.0.0.0/v1/vods", nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)

	return w
}

func TestDeleteVods(t *testing.T) {
	Convey("DeleteVods", t, func() {
		Convey("Fails when vod_ids is missing", func() {
			router := mockServer()
			p := url.Values{}
			w := performDeleteVodsRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: ids")
		})
		Convey("Fails when given an empty list of ids", func() {
			router := mockServer()
			p := url.Values{
				"ids": []string{""},
			}
			w := performDeleteVodsRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: ids")
		})
		Convey("Fails when given invalid ids", func() {
			router := mockServer()
			p := url.Values{
				"ids": []string{"123,abc"},
			}
			w := performDeleteVodsRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "ids is not a number:")
		})
		Convey("Succeeds when given a valid vod id", func() {
			router := mockServer()
			p := url.Values{
				"ids": []string{"1234"},
			}
			mockBackend.On("UpdateVod", mock.Anything, mock.Anything, int64(1234), models.VodUpdateInput{
				Deleted: models.NullBool{Bool: true, Valid: true},
			}).Return(nil, nil)
			w := performDeleteVodsRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusNoContent)
		})
		Convey("fails when update fails", func() {
			router := mockServer()
			p := url.Values{
				"ids": []string{"1234"},
			}
			mockBackend.On("UpdateVod", mock.Anything, mock.Anything, int64(1234), models.VodUpdateInput{
				Deleted: models.NullBool{Bool: true, Valid: true},
			}).Return(nil, fmt.Errorf("sdfd"))
			w := performDeleteVodsRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("Succeeds when given valid ids", func() {
			router := mockServer()
			p := url.Values{
				"ids": []string{"1,2"},
			}
			mockBackend.On("UpdateVod", mock.Anything, mock.Anything, int64(1), models.VodUpdateInput{
				Deleted: models.NullBool{Bool: true, Valid: true},
			}).Return(nil, nil)
			mockBackend.On("UpdateVod", mock.Anything, mock.Anything, int64(2), models.VodUpdateInput{
				Deleted: models.NullBool{Bool: true, Valid: true},
			}).Return(nil, nil)
			w := performDeleteVodsRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusNoContent)
		})
		Convey("fails when delete fails", func() {
			router := mockServer()
			vodIDs := []int64{123, 456}
			mockBackend.On("DeleteVods", mock.Anything, vodIDs).Return(fmt.Errorf("adf"))
			p := url.Values{
				"ids":         []string{"123,456"},
				"destructive": []string{"true"},
			}
			w := performDeleteVodsRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})
		Convey("Succeeds when destructive is included", func() {
			router := mockServer()
			vodIDs := []int64{123, 456}
			mockBackend.On("DeleteVods", mock.Anything, vodIDs).Return(nil)
			p := url.Values{
				"ids":         []string{"123,456"},
				"destructive": []string{"true"},
			}
			w := performDeleteVodsRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusNoContent)
		})
	})
}
