package api

import (
	"encoding/json"
	"io/ioutil"
	"net/http"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"

	"golang.org/x/net/context"
)

// FollowedMaxOffset is the max offset that this endpoint can query for
var FollowedMaxOffset = 1000

// Followed returns a set of VODs amongst a set of users that is followed.
func (s *Server) Followed(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	queryValues := r.URL.Query()

	broadcastType, err := setBroadcastType(queryValues.Get("broadcast_type"))
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	language := commaDelimited(queryValues.Get("language"))
	status := commaDelimited(queryValues.Get("status"))

	sort := queryValues.Get("sort")
	if sort != "views" {
		sort = "time"
	}

	limit, offset, err := pagination(queryValues.Get("limit"), queryValues.Get("offset"), FollowedMaxOffset)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	bodyBytes, err := ioutil.ReadAll(r.Body)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Could not read body: " + err.Error()})
		return
	}

	defer func() {
		closeErr := r.Body.Close()
		if closeErr != nil {
		}
	}()

	var followedIDs []int
	err = json.Unmarshal(bodyBytes, &followedIDs)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Invalid body: " + err.Error()})
		return
	}

	if len(followedIDs) == 0 {
		s.serveJSON(w, r, []*models.Vod{})
		return
	}

	vods, err := s.Backend.Followed(ctx, followedIDs, broadcastType, language, status, sort, limit, offset)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	s.serveJSON(w, r, vods)
}
