package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performFollowedRequest(router *Server, params url.Values, requestBody interface{}) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	body, err := json.Marshal(requestBody)
	So(err, ShouldBeNil)
	r, err := http.NewRequest("POST", "http://localhost/v1/vods/followed", bytes.NewBuffer(body))
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)

	return w
}

func TestFollowedVods(t *testing.T) {
	Convey("When sending reqeusts to /followed", t, func() {
		p := url.Values{
			"limit":  []string{"20"},
			"offset": []string{"0"},
		}

		Convey("request should fail without a broadcast type", func() {
			router := mockServer()
			w := performFollowedRequest(router, p, "")
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid value for broadcast_type")
		})

		Convey("request should fail with a json invalid body", func() {
			router := mockServer()
			p.Set("broadcast_type", "all")
			bogusBody := "4head"
			w := performFollowedRequest(router, p, bogusBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid body")
		})

		Convey("request pass with no follow ids", func() {
			router := mockServer()
			p.Set("broadcast_type", "all")
			emptyFollowIDs := []int{}
			w := performFollowedRequest(router, p, emptyFollowIDs)
			So(w.Code, ShouldEqual, http.StatusOK)
		})

		Convey("request should fail if backend fails", func() {
			router := mockServer()
			p.Set("broadcast_type", "all")
			body := []int{234, 3894}
			mockBackend.On("Followed", mock.Anything, body, []string{"archive", "highlight"}, []string{}, []string{}, "time", 20, 0).Return(nil, fmt.Errorf("sdf"))
			w := performFollowedRequest(router, p, body)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("request should succeed with a valid body", func() {
			router := mockServer()
			p.Set("broadcast_type", "all")
			p.Set("language", "blah")
			p.Set("sort", "views")
			body := []int{234, 3894}
			mockBackend.On("Followed", mock.Anything, body, []string{"archive", "highlight"}, []string{"blah"}, []string{}, "views", 20, 0).Return(nil, nil)
			w := performFollowedRequest(router, p, body)
			So(w.Code, ShouldEqual, http.StatusOK)
		})
	})
}
