package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performGetUserVideoPrivacyProperties(router *Server, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("GET", fmt.Sprintf("http://localhost/v1/user_video_privacy_properties/%v", params.Get("userID")), nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)
	return w
}

func TestGetUserVideoPrivacyProperties(t *testing.T) {
	var p url.Values
	Convey("GetUserVideoPrivacyProperties", t, func() {
		p = url.Values{}

		Convey("fail with NaN userID", func() {
			router := mockServer()
			p.Set("userID", "hithere")
			w := performGetUserVideoPrivacyProperties(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "user_id is not a number: hithere")
		})

		Convey("fails if backend fails", func() {
			router := mockServer()
			mockBackend.On("GetUserVideoPrivacyProperties", mock.Anything, int64(123)).Return(nil, fmt.Errorf("sdfsdf"))
			p.Set("userID", "123")
			w := performGetUserVideoPrivacyProperties(router, p)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("succeed with integer userID", func() {
			router := mockServer()
			mockBackend.On("GetUserVideoPrivacyProperties", mock.Anything, int64(123)).Return(nil, nil)
			p.Set("userID", "123")
			w := performGetUserVideoPrivacyProperties(router, p)
			So(w.Code, ShouldEqual, http.StatusOK)
			So(w.Body.String(), ShouldContainSubstring, "properties")
		})
	})
}
