package api

import (
	"net/http"

	"golang.org/x/net/context"

	"code.justin.tv/vod/vinyl/errors"
)

// GetVodAppealsMaxOffset is the max offset that this endpoint can query for
var GetVodAppealsMaxOffset = 0

// GetVodAppeals supports fetching vod appeals with the optional params listed in
// getVodAppealsOptionalParams
func (s *Server) GetVodAppeals(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	queryValues := r.URL.Query()
	priority, err := validateNullBool(queryValues.Get("priority"))
	if err != nil {
		s.serveError(ctx, w, r, errors.InvalidValueError{ParamName: "priority", Value: queryValues.Get("priority")})
		return
	}

	resolved, err := validateNullBool(queryValues.Get("resolved"))
	if err != nil {
		s.serveError(ctx, w, r, errors.InvalidValueError{ParamName: "resolved", Value: queryValues.Get("resolved")})
		return
	}

	userInfo := queryValues.Get("user_info")
	vodID, err := validateNullInt64(queryValues.Get("vod_id"))
	if err != nil {
		s.serveError(ctx, w, r, errors.IntegerParseError{ParamName: "vod_id", ParamValue: queryValues.Get("vod_id")})
		return
	}

	limit, offset, err := pagination(queryValues.Get("limit"), queryValues.Get("offset"), GetVodAppealsMaxOffset)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	appeals, err := s.Backend.GetVodAppeals(ctx, priority, resolved, userInfo, vodID, limit, offset)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	count, err := s.Backend.GetVodAppealsCount(ctx, priority, resolved, userInfo, vodID)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	ret := map[string]interface{}{
		"total_count": count,
		"appeals":     appeals,
	}

	s.serveJSON(w, r, ret)
}
