package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	"code.justin.tv/vod/vinyl/models"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performGetVodAppealsRequest(router *Server, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("GET", "http://localhost/v1/vod_appeals", nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)
	return w
}

func TestGetVodAppeals(t *testing.T) {
	Convey("GetVodAppeals", t, func() {
		Convey("with params", func() {
			Convey("fails for backend failure", func() {
				router := mockServer()
				p := url.Values{
					"priority": []string{"true"},
					"resolved": []string{"false"},
					"vod_id":   []string{"123"},
					"offset":   []string{"10"},
				}
				mockBackend.On("GetVodAppeals",
					mock.Anything,
					models.NullBool{Bool: true, Valid: true},
					models.NullBool{Bool: false, Valid: true},
					"",
					models.NullInt64{Int64: int64(123), Valid: true},
					DefaultLimit,
					10,
				).Return(nil, fmt.Errorf("sdflkj"))
				w := performGetVodAppealsRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusInternalServerError)
			})

			Convey("succeeds", func() {
				router := mockServer()
				p := url.Values{
					"priority": []string{"true"},
					"resolved": []string{"false"},
					"vod_id":   []string{"123"},
					"offset":   []string{"10"},
				}
				mockBackend.On("GetVodAppeals",
					mock.Anything,
					models.NullBool{Bool: true, Valid: true},
					models.NullBool{Bool: false, Valid: true},
					"",
					models.NullInt64{Int64: int64(123), Valid: true},
					DefaultLimit,
					10,
				).Return(nil, nil)
				mockBackend.On("GetVodAppealsCount",
					mock.Anything,
					models.NullBool{Bool: true, Valid: true},
					models.NullBool{Bool: false, Valid: true},
					"",
					models.NullInt64{Int64: int64(123), Valid: true},
				).Return(int64(0), nil)
				w := performGetVodAppealsRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusOK)
			})
		})

		Convey("should fail with an invalid", func() {
			Convey("limit", func() {
				router := mockServer()
				p := url.Values{"limit": []string{"1234"}}
				w := performGetVodAppealsRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "Invalid value for")
			})

			Convey("priority", func() {
				router := mockServer()
				p := url.Values{"priority": []string{"Mallory"}}
				w := performGetVodAppealsRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "Invalid value for")
			})

			Convey("resolved", func() {
				router := mockServer()
				p := url.Values{"resolved": []string{"Eve"}}
				w := performGetVodAppealsRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "Invalid value for")
			})

			Convey("vod_id", func() {
				router := mockServer()
				p := url.Values{"vod_id": []string{"Bob"}}
				w := performGetVodAppealsRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "is not a number")
			})
		})
	})
}
