package api

import (
	"net/http"
	"strconv"

	"goji.io/pat"

	"code.justin.tv/common/goauthorization"
	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"

	"golang.org/x/net/context"
)

// GetVodByID get VOD info
func (s *Server) GetVodByID(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	// Validate vod_id
	vodIDStr := pat.Param(ctx, "vod_id")
	if vodIDStr == "" {
		s.serveError(ctx, w, r, errors.MissingParamError{ParamName: "vod_id"})
		return
	}
	var vodID int64
	vodID, err := strconv.ParseInt(vodIDStr, 10, 64)
	if err != nil {
		s.serveError(ctx, w, r, errors.IntegerParseError{ParamName: "vod_id", ParamValue: vodIDStr})
		return
	}

	// Validate parameters
	queryValues := r.URL.Query()

	appealsAndAMRs, err := validateBool(queryValues.Get("appeals_and_amrs"))
	if err != nil {
		s.serveError(ctx, w, r, errors.InvalidValueError{ParamName: "appeals_and_amrs", Value: queryValues.Get("appeals_and_amrs")})
		return
	}

	notificationSettings, err := validateBool(queryValues.Get("notification_settings"))
	if err != nil {
		s.serveError(ctx, w, r, errors.InvalidValueError{ParamName: "notification_settings", Value: queryValues.Get("notification_settings")})
		return
	}

	filters, err := models.ParseFilters(queryValues)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	// Load VOD from backend
	vods, err := s.Backend.GetVodsByID(ctx, []int64{int64(vodID)}, appealsAndAMRs, notificationSettings, filters)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}
	if len(vods) == 0 {
		w.Header().Add("Cache-Control", "public, max-age=60")
		s.serveError(ctx, w, r, errors.NotFoundError{Type: "Vod", ID: int(vodID)})
		return
	}
	vod := vods[0]

	// Authorization: check JWT token, only needed if the VOD is not public
	if !vod.Watchable() {
		manageVodErr := s.authHandler.AuthorizeJWT(r, goauthorization.CapabilityClaims{
			"manage_vod": goauthorization.CapabilityClaim{
				"vod_id": vodIDStr,
			},
		})
		manageAppealErr := s.authHandler.AuthorizeJWT(r, goauthorization.CapabilityClaims{
			"manage_appeals": goauthorization.CapabilityClaim{
				"vod_id": vodIDStr,
			},
		})

		if manageVodErr != nil && manageAppealErr != nil {
			s.serveError(ctx, w, r, errors.UnauthorizedError{})
			return
		}
	}

	w.Header().Add("Cache-Control", "public, max-age=60")
	// Respond JSON
	s.serveJSON(w, r, vod)
}
