package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	"code.justin.tv/common/config"
	"code.justin.tv/vod/vinyl/api/mocks"
	"code.justin.tv/vod/vinyl/auth"
	"code.justin.tv/vod/vinyl/models"

	"github.com/cactus/go-statsd-client/statsd"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performGetVodByIDRequest(router *Server, vodIDStr string, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("GET", fmt.Sprintf("http://localhost/v1/vods/%s", vodIDStr), nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)
	return w
}

func TestGetVodByID(t *testing.T) {
	var p url.Values
	Convey("GetVodsByID", t, func() {
		p = url.Values{}

		Convey("fails when the :vod_id is invalid", func() {
			router := mockServer()
			w := performGetVodByIDRequest(router, "invalidID", p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "id is not a number: invalidID")
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})

		Convey("fails when appeals_and_amrs invalid", func() {
			router := mockServer()
			p.Set("appeals_and_amrs", "whatever")
			w := performGetVodByIDRequest(router, "123", p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})

		Convey("fails when notification_settings invalid", func() {
			router := mockServer()
			p.Set("notification_settings", "whatever")
			w := performGetVodByIDRequest(router, "123", p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})

		Convey("fails when include_deleted invalid", func() {
			router := mockServer()
			p.Set("include_deleted", "whatever")
			w := performGetVodByIDRequest(router, "123", p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})

		Convey("backend error", func() {
			router := mockServer()
			mockBackend.On("GetVodsByID",
				mock.Anything,
				[]int64{int64(123)},
				false,
				false,
				&models.VodFilters{
					IncludeDeleted:    true,
					IncludePrivate:    true,
					IncludeProcessing: true,
				},
			).Return(nil, fmt.Errorf("Sdfsdf"))
			w := performGetVodByIDRequest(router, "123", p)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})

		Convey("no results", func() {
			router := mockServer()
			mockBackend.On("GetVodsByID",
				mock.Anything,
				[]int64{int64(123)},
				false,
				false,
				&models.VodFilters{
					IncludeDeleted:    true,
					IncludePrivate:    true,
					IncludeProcessing: true,
				},
			).Return([]*models.Vod{}, nil)
			w := performGetVodByIDRequest(router, "123", p)
			So(w.Code, ShouldEqual, http.StatusNotFound)
			So(w.Header().Get("Cache-Control"), ShouldEqual, "public, max-age=60")
		})

		Convey("succeeds with valid id", func() {
			router := mockServer()
			mockBackend.On("GetVodsByID",
				mock.Anything,
				[]int64{int64(123)},
				false,
				false,
				&models.VodFilters{
					IncludeDeleted:    true,
					IncludePrivate:    true,
					IncludeProcessing: false,
				},
			).Return([]*models.Vod{&models.Vod{Status: "sdfh"}}, nil)

			p.Set("include_processing", "false")
			w := performGetVodByIDRequest(router, "123", p)
			So(w.Code, ShouldEqual, http.StatusOK)
			So(w.Header().Get("Cache-Control"), ShouldEqual, "public, max-age=60")
		})

		Convey("fails when auth fails", func() {
			// reconfigure the request object to fake an auth failure
			w := httptest.NewRecorder()
			stats, _ := statsd.NewNoopClient()
			logger := config.RollbarErrorLogger()
			noopBackend := &mocks.Backend{}
			noopBackend.On("GetVodsByID",
				mock.Anything,
				[]int64{int64(123)},
				false,
				false,
				&models.VodFilters{
					IncludeDeleted:    true,
					IncludePrivate:    true,
					IncludeProcessing: true,
				},
			).Return([]*models.Vod{&models.Vod{Status: "sdfh"}}, nil)
			authThatFails := auth.NewFakeHandler(false)
			server, _ := NewServer(stats, logger, noopBackend, authThatFails)

			r, err := http.NewRequest("GET", "http://localhost/v1/vods/123", nil)
			So(err, ShouldBeNil)
			p.Set("ids", "234")
			r.URL.RawQuery = p.Encode()
			server.ServeHTTP(w, r)

			// Now check that the status is Unauthorized
			So(w.Code, ShouldEqual, http.StatusUnauthorized)
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})
	})
}
