package api

import (
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performGetVodsAggregationByIDRequest(router *Server, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("GET", "http://localhost/v1/vods_aggregation", nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)
	return w
}

func TestGetVodsAggregationByID(t *testing.T) {
	var p url.Values
	Convey("GetVodsAggregationByID", t, func() {
		p = url.Values{}

		Convey("should fail without the `ids` parameter", func() {
			router := mockServer()
			w := performGetVodsAggregationByIDRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: ids")
		})

		Convey("should fail when the `ids` paramter contains invalid values", func() {
			router := mockServer()
			p.Set("ids", "foobar")
			w := performGetVodsAggregationByIDRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "ids is not a number: foobar")
		})

		Convey("should fail when include_deleted is invalid", func() {
			router := mockServer()
			p.Set("ids", "234,23498,23093")
			p.Set("include_deleted", "bananas")
			w := performGetVodsAggregationByIDRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
		})

		Convey("should suceed when integers are passed as ids", func() {
			router := mockServer()
			mockBackend.On("GetVodsAggregationByID",
				mock.Anything,
				[]int64{int64(234), int64(23498)},
				&models.VodFilters{
					IncludeDeleted:    true,
					IncludePrivate:    true,
					IncludeProcessing: true,
				},
			).Return(&models.VodsAggregation{}, nil)
			p.Set("ids", "234,23498")
			w := performGetVodsAggregationByIDRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusOK)
		})
	})
}
