package api

import (
	"net/http"
	"strconv"
	"strings"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"

	"golang.org/x/net/context"
)

// GetVodsByID supports fetching VOD information by VOD ID
func (s *Server) GetVodsByID(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	queryValues := r.URL.Query()

	vodIDsStr := queryValues.Get("ids")
	if vodIDsStr == "" {
		s.serveError(ctx, w, r, errors.MissingParamError{ParamName: "ids"})
		return
	}

	appealsAndAMRs, err := validateBool(queryValues.Get("appeals_and_amrs"))
	if err != nil {
		s.serveError(ctx, w, r, errors.InvalidValueError{ParamName: "appeals_and_amrs", Value: queryValues.Get("appeals_and_amrs")})
		return
	}

	notificationSettings, err := validateBool(queryValues.Get("notification_settings"))
	if err != nil {
		s.serveError(ctx, w, r, errors.InvalidValueError{ParamName: "notification_settings", Value: queryValues.Get("notification_settings")})
		return
	}

	filters, err := models.ParseFilters(queryValues)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	vodIDs, err := toInt64List(vodIDsStr)
	if err != nil {
		s.serveError(ctx, w, r, errors.IntegerParseError{ParamName: "ids", ParamValue: vodIDsStr})
		return
	}

	vods, err := s.Backend.GetVodsByID(ctx, vodIDs, appealsAndAMRs, notificationSettings, filters)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	ret := map[string]interface{}{
		"vods": vods,
	}

	w.Header().Add("Cache-Control", "public, max-age=60")
	s.serveJSON(w, r, ret)
}

// Parse comma-separated string into a list of int64 vodIDs
func toInt64List(vodIDsStr string) ([]int64, error) {
	vodIDsStrList := strings.Split(vodIDsStr, ",")
	vodIDs := make([]int64, len(vodIDsStrList))
	for i, vodIDStr := range vodIDsStrList {
		vodID, err := strconv.ParseInt(vodIDStr, 10, 32)
		if err != nil {
			return nil, err
		}
		vodIDs[i] = int64(vodID)
	}
	return vodIDs, nil
}
