package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performGetVodsByIDRequest(router *Server, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("GET", "http://localhost/v1/vods", nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)
	return w
}

func TestGetVodsByID(t *testing.T) {
	var p url.Values
	Convey("GetVodsByID", t, func() {
		p = url.Values{}

		Convey("should fail without the `ids` parameter", func() {
			router := mockServer()
			w := performGetVodsByIDRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: ids")
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})

		Convey("should fail when the `ids` paramter contains invalid values", func() {
			router := mockServer()
			p.Set("ids", "foobar")
			w := performGetVodsByIDRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "ids is not a number: foobar")
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})

		Convey("should fail when appeals_and_amrs invalid", func() {
			router := mockServer()
			p.Set("ids", "foobar")
			p.Set("appeals_and_amrs", "sdfdsf")
			w := performGetVodsByIDRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
		})

		Convey("should fail when notification_settings invalid", func() {
			router := mockServer()
			p.Set("ids", "foobar")
			p.Set("notification_settings", "sdfdsf")
			w := performGetVodsByIDRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
		})

		Convey("should fail when backend error", func() {
			router := mockServer()
			mockBackend.On("GetVodsByID",
				mock.Anything,
				[]int64{int64(1), int64(2)},
				false,
				false,
				&models.VodFilters{
					IncludeDeleted:    true,
					IncludePrivate:    true,
					IncludeProcessing: true,
				},
			).Return(nil, fmt.Errorf("what"))
			p.Set("ids", "1,2")
			w := performGetVodsByIDRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("should suceed with parameters", func() {
			router := mockServer()
			mockBackend.On("GetVodsByID",
				mock.Anything,
				[]int64{int64(1), int64(2)},
				true,
				false,
				&models.VodFilters{
					IncludeDeleted:    true,
					IncludePrivate:    true,
					IncludeProcessing: false,
				},
			).Return(nil, nil)
			p.Set("ids", "1,2")
			p.Set("appeals_and_amrs", "true")
			p.Set("include_processing", "false")
			w := performGetVodsByIDRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusOK)
			So(w.Body.String(), ShouldContainSubstring, "vods")
			So(w.Header().Get("Cache-Control"), ShouldEqual, "public, max-age=60")
		})

		Convey("should fail when include_deleted is invalid", func() {
			router := mockServer()
			p.Set("ids", "234,23498,23093")
			p.Set("include_deleted", "bananas")
			w := performGetVodsByIDRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})
	})
}
