package api

import (
	"net/http"
	"strconv"
	"time"

	"goji.io/pat"
	"golang.org/x/net/context"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"
)

var (
	// validParamValues stores the valid values for status
	validParamValues = map[string]map[string]bool{
		"status": map[string]bool{
			models.StatusRecording:   true,
			models.StatusRecorded:    true,
			models.StatusUnprocessed: true,
		},
	}
)

// GetVodsByStatus supports fetching VODs with a specific status (required),
// broadcast type (optional), and a time frame (optional).
func (s *Server) GetVodsByStatus(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	status := pat.Param(ctx, "status")
	if status == "" {
		s.serveError(ctx, w, r, errors.MissingParamError{ParamName: "params"})
		return
	}

	if !isValidParamValue("status", status) {
		s.serveError(ctx, w, r, errors.InvalidValueError{ParamName: "status", Value: status})
		return
	}

	// pull out optional params here
	queryValues := r.URL.Query()
	b := queryValues.Get("broadcast_type")
	if b == "" { // specifically for this endpoint, default broadcast_type to all
		b = "all"
	}
	broadcastType, err := setBroadcastType(b)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	startTimeString := queryValues.Get("start_time")
	startTime, err := stringToTime(startTimeString)
	if err != nil {
		s.serveError(ctx, w, r, errors.InvalidValueError{ParamName: "start_time", Value: startTimeString})
		return
	}

	endTimeString := queryValues.Get("end_time")
	endTime, err := stringToTime(endTimeString)
	if err != nil {
		s.serveError(ctx, w, r, errors.InvalidValueError{ParamName: "end_time", Value: endTimeString})
		return
	}

	vods, err := s.Backend.GetVodsByStatus(ctx, status, broadcastType, startTime, endTime)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	s.serveJSON(w, r, vods)
}

// isValidParamValue returns whether or not a value is a valid param value
func isValidParamValue(paramName, value string) bool {
	_, ok := validParamValues[paramName][value]
	return ok
}

// stringToTime will convert a string to a time.Time struct. if "" is given, it
// will return the zero time and no error because time is an optional param. note
// that UTC is forced to make sure it is not in local time
func stringToTime(secondsString string) (time.Time, error) {
	if secondsString == "" {
		return time.Time{}, nil
	}

	seconds, err := strconv.ParseInt(secondsString, 10, 64)
	if err == nil {
		return time.Unix(seconds, int64(0)).UTC(), nil
	}

	return time.Time{}, err

}
