package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"
	"time"

	"code.justin.tv/vod/vinyl/models"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performGetVodsByStatusRequest(router *Server, status string, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("GET", fmt.Sprintf("http://localhost/v1/vods/status/%s", status), nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)
	return w
}

func TestGetVodsByStatus(t *testing.T) {
	Convey("GetVodsByStatus", t, func() {
		Convey("should succeed with status", func() {
			Convey("and no broadcast type or time frame", func() {
				router := mockServer()
				p := url.Values{}
				mockBackend.On("GetVodsByStatus", mock.Anything, models.StatusRecording, []string{"archive", "highlight"}, time.Time{}, time.Time{}).Return(nil, nil)
				w := performGetVodsByStatusRequest(router, models.StatusRecording, p)
				So(w.Code, ShouldEqual, http.StatusOK)
			})

			Convey("and a broadcast type", func() {
				router := mockServer()
				p := url.Values{"broadcast_type": []string{"highlight"}}
				mockBackend.On("GetVodsByStatus", mock.Anything, models.StatusRecording, []string{"highlight"}, time.Time{}, time.Time{}).Return(nil, nil)
				w := performGetVodsByStatusRequest(router, models.StatusRecording, p)
				So(w.Code, ShouldEqual, http.StatusOK)
			})

			Convey("and a start time", func() {
				router := mockServer()
				start := int64(1465419990)
				p := url.Values{"start_time": []string{strconv.FormatInt(start, 10)}}
				mockBackend.On("GetVodsByStatus", mock.Anything, models.StatusRecording, []string{"archive", "highlight"}, time.Unix(start, 0).UTC(), time.Time{}).Return(nil, nil)
				w := performGetVodsByStatusRequest(router, models.StatusRecording, p)
				So(w.Code, ShouldEqual, http.StatusOK)
			})

			Convey("and an end time", func() {
				router := mockServer()
				end := int64(1465419990)
				p := url.Values{"end_time": []string{strconv.FormatInt(end, 10)}}
				mockBackend.On("GetVodsByStatus", mock.Anything, models.StatusRecording, []string{"archive", "highlight"}, time.Time{}, time.Unix(end, 0).UTC()).Return(nil, nil)
				w := performGetVodsByStatusRequest(router, models.StatusRecording, p)
				So(w.Code, ShouldEqual, http.StatusOK)
			})
		})

		Convey("should fail with ", func() {
			Convey("backend error", func() {
				router := mockServer()
				p := url.Values{}
				mockBackend.On("GetVodsByStatus", mock.Anything, models.StatusRecording, []string{"archive", "highlight"}, time.Time{}, time.Time{}).Return(nil, fmt.Errorf("sdfd"))
				w := performGetVodsByStatusRequest(router, models.StatusRecording, p)
				So(w.Code, ShouldEqual, http.StatusInternalServerError)
			})

			Convey("an invalid status", func() {
				router := mockServer()
				p := url.Values{}
				w := performGetVodsByStatusRequest(router, "invalid status", p)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "Invalid value")
			})

			Convey("an invalid broadcast type", func() {
				router := mockServer()
				p := url.Values{"broadcast_type": []string{"invalid broadcast_type"}}
				w := performGetVodsByStatusRequest(router, models.StatusRecording, p)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "Invalid value")
			})

			Convey("an invalid start time", func() {
				router := mockServer()
				p := url.Values{"start_time": []string{"invalid start_time"}}
				w := performGetVodsByStatusRequest(router, models.StatusRecording, p)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "Invalid value")
			})

			Convey("an invalid end time", func() {
				router := mockServer()
				p := url.Values{"end_time": []string{"invalid end_time"}}
				w := performGetVodsByStatusRequest(router, models.StatusRecording, p)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "Invalid value")
			})
		})
	})
}
