package api

import (
	"net/http"
	"strconv"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"

	"goji.io/pat"
	"golang.org/x/net/context"
)

// GetVodsByUserMaxOffset is the max offset that this endpoint can query for
var GetVodsByUserMaxOffset = 0

// GetVodsByUser supports fetching VOD information for a specific user by broadcast_type.
func (s *Server) GetVodsByUser(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "user_id")
	if userID == "" {
		s.serveError(ctx, w, r, errors.MissingParamError{ParamName: "user_id"})
		return
	}
	user, err := strconv.Atoi(userID)
	if err != nil {
		s.serveError(ctx, w, r, errors.IntegerParseError{ParamName: "user_id", ParamValue: userID})
		return
	}

	queryValues := r.URL.Query()

	limit, offset, err := pagination(queryValues.Get("limit"), queryValues.Get("offset"), GetVodsByUserMaxOffset)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	b := queryValues.Get("broadcast_type")
	if b == "" { // specifically for this endpoint, default broadcast_type to highlight
		b = "highlight"
	}
	broadcastType, err := setBroadcastType(b)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	language := commaDelimited(queryValues.Get("language"))
	status := commaDelimited(queryValues.Get("status"))

	sort := queryValues.Get("sort")
	if sort != "views" {
		sort = "time"
	}

	appealsAndAMRs, err := validateBool(queryValues.Get("appeals_and_amrs"))
	if err != nil {
		s.serveError(ctx, w, r, errors.InvalidValueError{ParamName: "appeals_and_amrs", Value: queryValues.Get("appeals_and_amrs")})
		return
	}

	notificationSettings, err := validateBool(queryValues.Get("notification_settings"))
	if err != nil {
		s.serveError(ctx, w, r, errors.InvalidValueError{ParamName: "notification_settings", Value: queryValues.Get("notification_settings")})
		return
	}

	filters, err := models.ParseFilters(queryValues)
	if err != nil {
		s.serveError(ctx, w, r, err)
	}

	vods, err := s.Backend.GetVodsByUser(ctx, user, broadcastType, language, status, appealsAndAMRs, notificationSettings, filters, sort, limit, offset)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	count, err := s.Backend.GetVodsCountByUser(ctx, user, broadcastType, language, status, filters)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	ret := map[string]interface{}{
		"vods":        vods,
		"total_count": count,
	}

	w.Header().Add("Cache-Control", "public, max-age=300")
	s.serveJSON(w, r, ret)
}
