package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performGetVodsByUserRequest(router *Server, userID interface{}, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("GET", fmt.Sprintf("http://localhost/v1/vods/user/%v", userID), nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)
	return w
}

func TestGetVodsByUser(t *testing.T) {
	Convey("GetVodsByUser", t, func() {
		Convey("should fail when the userID sent is not an integer", func() {
			router := mockServer()
			p := url.Values{}
			w := performGetVodsByUserRequest(router, "cow", p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "user_id is not a number: cow")
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})

		Convey("should fail when the broadcast_type sent is not valid", func() {
			router := mockServer()
			p := url.Values{}
			p.Set("broadcast_type", "stuff")
			w := performGetVodsByUserRequest(router, 123, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid value for broadcast_type: stuff")
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})

		Convey("should fail when the `limit` parameter contains invalid values", func() {
			router := mockServer()
			p := url.Values{}
			p.Set("limit", "foobar")
			w := performGetVodsByUserRequest(router, 123, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "limit is not a number: foobar")
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})

		Convey("should fail when the `offset` parameter contains invalid values", func() {
			router := mockServer()
			p := url.Values{}
			p.Set("offset", "foobar")
			w := performGetVodsByUserRequest(router, 123, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "offset is not a number: foobar")
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})

		Convey("should fail when the `appeals_and_amrs` parameter invalid", func() {
			router := mockServer()
			p := url.Values{}
			p.Set("appeals_and_amrs", "foobar")
			w := performGetVodsByUserRequest(router, 123, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})

		Convey("should fail when the `notification_settings` parameter invalid", func() {
			router := mockServer()
			p := url.Values{}
			p.Set("notification_settings", "foobar")
			w := performGetVodsByUserRequest(router, 123, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})

		Convey("should fail for backend error", func() {
			router := mockServer()
			mockBackend.On("GetVodsByUser",
				mock.Anything,
				123,
				[]string{"upload"},
				[]string{},
				[]string{},
				false,
				false,
				&models.VodFilters{
					IncludeDeleted:    true,
					IncludePrivate:    true,
					IncludeProcessing: true,
				},
				"views",
				5,
				123,
			).Return(nil, fmt.Errorf("what"))

			p := url.Values{}
			p.Set("offset", "123")
			p.Set("sort", "views")
			p.Set("limit", "5")
			p.Set("broadcast_type", "upload")
			w := performGetVodsByUserRequest(router, 123, p)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
			So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
		})

		Convey("should succeed", func() {
			router := mockServer()
			mockBackend.On("GetVodsByUser",
				mock.Anything,
				123,
				[]string{"upload"},
				[]string{},
				[]string{},
				false,
				false,
				&models.VodFilters{
					IncludeDeleted:    true,
					IncludePrivate:    true,
					IncludeProcessing: false,
				},
				"time",
				5,
				123,
			).Return(nil, nil)
			mockBackend.On("GetVodsCountByUser",
				mock.Anything,
				123,
				[]string{"upload"},
				[]string{},
				[]string{},
				&models.VodFilters{
					IncludeDeleted:    true,
					IncludePrivate:    true,
					IncludeProcessing: false,
				},
			).Return(int64(0), nil)

			p := url.Values{}
			p.Set("offset", "123")
			p.Set("limit", "5")
			p.Set("broadcast_type", "upload")
			p.Set("include_processing", "false")
			w := performGetVodsByUserRequest(router, 123, p)
			So(w.Code, ShouldEqual, http.StatusOK)
			So(w.Header().Get("Cache-Control"), ShouldEqual, "public, max-age=300")
		})

	})
}
