package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performModerationRequest(router *Server, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("GET", "http://localhost/v1/vods/moderation/88766660", nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)
	return w
}

func TestModeration(t *testing.T) {
	Convey("ModerationCallback", t, func() {
		p := url.Values{}

		Convey("should fail without the `status` parameter", func() {
			router := mockServer()
			w := performModerationRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Missing required param: status")
		})

		Convey("should fail when the `status` parameter contains invalid values", func() {
			router := mockServer()
			p.Set("status", "fail")
			w := performModerationRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "status is not a number: fail")
		})

		Convey("should fail when status is not an integer 1 to 4", func() {
			router := mockServer()
			p.Set("status", "10")
			w := performModerationRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid value for status:")
		})

		Convey("should fail when backend error", func() {
			router := mockServer()
			mockBackend.On("Moderation", mock.Anything, int64(88766660), 1).Return(fmt.Errorf("what"))
			p.Set("status", "1")
			w := performModerationRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("should succeed when status is an integer 1 to 4", func() {
			router := mockServer()
			mockBackend.On("Moderation", mock.Anything, int64(88766660), 1).Return(nil)
			p.Set("status", "1")
			w := performModerationRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusNoContent)
		})
	})
}
