package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performOwnerForRechatRequest(router *Server, vodIDStr string, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("GET", fmt.Sprintf("http://localhost/v1/vods/%s/owner_for_rechat", vodIDStr), nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)
	return w
}
func TestOwnerForRechat(t *testing.T) {
	var p url.Values
	Convey("OwnerForRechat", t, func() {
		p = url.Values{}

		Convey("fails when the :vod_id is invalid", func() {
			router := mockServer()
			w := performOwnerForRechatRequest(router, "invalidID", p)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "vod_id is not a number: invalidID")
		})

		Convey("fails with backend error", func() {
			router := mockServer()
			mockBackend.On("OwnerForRechat",
				mock.Anything,
				int64(123),
			).Return(nil, fmt.Errorf("Sdfsdf"))
			w := performOwnerForRechatRequest(router, "123", p)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("fails when cannot find vod", func() {
			router := mockServer()
			mockBackend.On("OwnerForRechat",
				mock.Anything,
				int64(123),
			).Return(nil, errors.NotFoundError{Type: "Vod", ID: int(123)})
			w := performOwnerForRechatRequest(router, "123", p)
			So(w.Code, ShouldEqual, http.StatusNotFound)
		})

		Convey("succeeds with valid id", func() {
			router := mockServer()
			mockBackend.On("OwnerForRechat",
				mock.Anything,
				int64(123),
			).Return(&models.VodOwner{OwnerID: "lala"}, nil)

			w := performOwnerForRechatRequest(router, "123", p)
			So(w.Code, ShouldEqual, http.StatusOK)
		})
	})
}
