package api

import (
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

func TestPagination(t *testing.T) {
	var maxOffset int
	Convey("limit", t, func() {
		offsetQueryValue := "0"
		maxOffset = 0
		Convey("acceptable", func() {
			limitQueryValue := "20"
			limit, offset, err := pagination(limitQueryValue, offsetQueryValue, maxOffset)
			So(limit, ShouldEqual, 20)
			So(offset, ShouldEqual, 0)
			So(err, ShouldBeNil)
		})
		Convey("NaN", func() {
			limitQueryValue := "starcraft"
			_, _, err := pagination(limitQueryValue, offsetQueryValue, maxOffset)
			So(err.Error(), ShouldContainSubstring, "limit is not a number")
		})
		Convey("negative", func() {
			limitQueryValue := "-1"
			_, _, err := pagination(limitQueryValue, offsetQueryValue, maxOffset)
			So(err.Error(), ShouldContainSubstring, "Invalid value for limit")
		})
		Convey("0 uses default", func() {
			limitQueryValue := "0"
			limit, offset, err := pagination(limitQueryValue, offsetQueryValue, maxOffset)
			So(limit, ShouldEqual, DefaultLimit)
			So(offset, ShouldEqual, 0)
			So(err, ShouldBeNil)
		})
		Convey("empty uses default", func() {
			limitQueryValue := ""
			limit, offset, err := pagination(limitQueryValue, offsetQueryValue, maxOffset)
			So(limit, ShouldEqual, DefaultLimit)
			So(offset, ShouldEqual, 0)
			So(err, ShouldBeNil)
		})
		Convey("greater than max errors", func() {
			limitQueryValue := "10000000"
			_, _, err := pagination(limitQueryValue, offsetQueryValue, maxOffset)
			So(err.Error(), ShouldContainSubstring, "Invalid value for limit")
		})
	})
	Convey("offset", t, func() {
		limitQueryValue := "20"
		maxOffset = 0
		Convey("acceptable", func() {
			offsetQueryValue := "60"
			limit, offset, err := pagination(limitQueryValue, offsetQueryValue, maxOffset)
			So(limit, ShouldEqual, 20)
			So(offset, ShouldEqual, 60)
			So(err, ShouldBeNil)
		})
		Convey("NaN", func() {
			offsetQueryValue := "league"
			_, _, err := pagination(limitQueryValue, offsetQueryValue, maxOffset)
			So(err.Error(), ShouldContainSubstring, "offset is not a number")
		})
		Convey("negative", func() {
			offsetQueryValue := "-1"
			_, _, err := pagination(limitQueryValue, offsetQueryValue, maxOffset)
			So(err.Error(), ShouldContainSubstring, "Invalid value for offset")
		})
		Convey("0 stays 0", func() {
			offsetQueryValue := "0"
			limit, offset, err := pagination(limitQueryValue, offsetQueryValue, maxOffset)
			So(limit, ShouldEqual, 20)
			So(offset, ShouldEqual, 0)
			So(err, ShouldBeNil)
		})
		Convey("empty becomes 0", func() {
			offsetQueryValue := ""
			limit, offset, err := pagination(limitQueryValue, offsetQueryValue, maxOffset)
			So(limit, ShouldEqual, 20)
			So(offset, ShouldEqual, 0)
			So(err, ShouldBeNil)
		})
		Convey("huge offset is fine", func() {
			offsetQueryValue := "1000000"
			limit, offset, err := pagination(limitQueryValue, offsetQueryValue, maxOffset)
			So(limit, ShouldEqual, 20)
			So(offset, ShouldEqual, 1000000)
			So(err, ShouldBeNil)
		})
		Convey("offset larger than max errors", func() {
			offsetQueryValue := "1000000"
			maxOffset = 1000
			_, _, err := pagination(limitQueryValue, offsetQueryValue, maxOffset)
			So(err.Error(), ShouldContainSubstring, "Invalid value for offset")
		})
	})
}
