package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performRelatedRequest(router *Server, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("GET", "http://localhost:8000/v1/vods/related", nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()
	router.ServeHTTP(w, r)
	return w
}

func TestRelated(t *testing.T) {
	var p url.Values
	userID := 123
	vodID := 456
	Convey("related", t, func() {
		p = url.Values{
			"user_id": []string{strconv.Itoa(userID)},
			"vod_id":  []string{strconv.Itoa(vodID)},
		}
		Convey("user_id", func() {
			Convey("valid", func() {
				router := mockServer()
				mockBackend.On("Related", mock.Anything, userID, vodID).Return(nil, nil)
				w := performRelatedRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusOK)
			})
			Convey("backend error", func() {
				router := mockServer()
				mockBackend.On("Related", mock.Anything, userID, vodID).Return(nil, fmt.Errorf("what"))
				w := performRelatedRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusInternalServerError)
			})
			Convey("invalid", func() {
				router := mockServer()
				p.Set("user_id", "")
				w := performRelatedRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "Missing required param: user_id")
			})
			Convey("malformed", func() {
				router := mockServer()
				p.Set("user_id", "sdfkldj")
				w := performRelatedRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "user_id is not a number")
			})
		})
		Convey("vod_id", func() {
			Convey("valid", func() {
				router := mockServer()
				mockBackend.On("Related", mock.Anything, userID, vodID).Return(nil, nil)
				w := performRelatedRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusOK)
			})
			Convey("malformed", func() {
				router := mockServer()
				p.Set("vod_id", "sdfdsf")
				w := performRelatedRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
			})
			Convey("missing is ok", func() {
				router := mockServer()
				mockBackend.On("Related", mock.Anything, userID, 0).Return(nil, nil)
				p.Set("vod_id", "")
				w := performRelatedRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusOK)
			})
			Convey("0 behaves the same as missing", func() {
				router := mockServer()
				mockBackend.On("Related", mock.Anything, userID, 0).Return(nil, nil)
				p.Set("vod_id", "0")
				w := performRelatedRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusOK)
			})
		})
	})
}
