package api

import (
	"encoding/json"

	"code.justin.tv/vod/vinyl/errors"
)

// AllowBodyFields takes in a byte aray (the JSON body of a request)
// and validates that the top level keys in the body are within a set of allowable keys.
func AllowBodyFields(input []byte, allowedKeys []string) error {
	genericMap := map[string]interface{}{}
	err := json.Unmarshal(input, &genericMap)
	if err != nil {
		return errors.RequestBodyError{ErrorString: "Invalid body: " + err.Error()}
	}

	for key := range genericMap {
		found := false
		for _, allowedKey := range allowedKeys {
			if key == allowedKey {
				found = true
				break
			}
		}
		if !found {
			return errors.DisallowedParamError{ParamName: key}
		}
	}

	return nil
}

// RequireFieldsForAllModels takes in a byte array (the JSON body of a request)
// and validates that it contains individual keys (fields) for each top level key.
func RequireFieldsForAllModels(input []byte, fieldMap map[string][]string) error {
	genericMap := map[string][]map[string]interface{}{}

	err := json.Unmarshal(input, &genericMap)
	if err != nil {
		return errors.RequestBodyError{ErrorString: "Invalid body: " + err.Error()}
	}

	for key, reqFields := range fieldMap {
		elements, ok := genericMap[key]
		if !ok {
			return errors.MissingParamError{ParamName: key}
		}

		for _, field := range reqFields {
			for _, el := range elements {
				_, ok := el[field]
				if !ok {
					return errors.MissingParamError{ParamName: field}
				}
			}
		}

	}

	return nil
}
