package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performResolveTrackAppealTest(router *Server, appealID interface{}, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("POST", fmt.Sprintf("http://localhost/v1/track_appeal/%v", appealID), nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)
	return w
}

func TestResolveTrackAppeal(t *testing.T) {
	Convey("When resolving track appeals", t, func() {
		params := url.Values{}
		Convey("request should fail without an integer appeal ID", func() {
			router := mockServer()
			w := performResolveTrackAppealTest(router, "cow", params)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "appeal_id is not a number")
		})

		Convey("request should fail without an action", func() {
			router := mockServer()
			w := performResolveTrackAppealTest(router, 123, params)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid value for action")
		})

		Convey("request should fail with an action that is not valid", func() {
			router := mockServer()
			params.Set("action", "hello")
			w := performResolveTrackAppealTest(router, 123, params)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid value for action: hello")
		})

		Convey("fail if backend fail", func() {
			router := mockServer()
			mockBackend.On("ResolveTrackAppeal", mock.Anything, int64(123), true).Return(fmt.Errorf("what"))
			params.Set("action", "accept")
			w := performResolveTrackAppealTest(router, 123, params)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("request should succeed with an accept action", func() {
			router := mockServer()
			mockBackend.On("ResolveTrackAppeal", mock.Anything, int64(123), true).Return(nil)
			params.Set("action", "accept")
			w := performResolveTrackAppealTest(router, 123, params)
			So(w.Code, ShouldEqual, http.StatusNoContent)
		})

		Convey("request should succeed with an reject action", func() {
			router := mockServer()
			params.Set("action", "reject")
			mockBackend.On("ResolveTrackAppeal", mock.Anything, int64(123), false).Return(nil)
			w := performResolveTrackAppealTest(router, 123, params)
			So(w.Code, ShouldEqual, http.StatusNoContent)
		})
	})
}
