package api

import (
	"encoding/json"
	"io/ioutil"
	"net/http"
	"strconv"

	"goji.io/pat"
	"golang.org/x/net/context"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"
)

// SetUserVideoPrivacyProperties sets video privacy properties for a user
func (s *Server) SetUserVideoPrivacyProperties(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "user_id")
	if userID == "" {
		s.serveError(ctx, w, r, errors.MissingParamError{ParamName: "user_id"})
		return
	}

	user, err := strconv.ParseInt(userID, 10, 64)
	if err != nil {
		s.serveError(ctx, w, r, errors.IntegerParseError{ParamName: "user_id", ParamValue: userID})
		return
	}

	bodyBytes, err := ioutil.ReadAll(r.Body)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Could not read body: " + err.Error()})
		return
	}

	defer func() {
		closeErr := r.Body.Close()
		if closeErr != nil {
		}
	}()

	err = AllowBodyFields(bodyBytes, models.UserVideoPrivacyPropertiesFields)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	u := &models.UserVideoPrivacyPropertiesInput{}
	err = json.Unmarshal(bodyBytes, &u)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Invalid body: " + err.Error()})
		return
	}

	p, err := s.Backend.SetUserVideoPrivacyProperties(ctx, user, u)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	ret := map[string]*models.UserVideoPrivacyProperties{
		"properties": p,
	}
	s.serveJSON(w, r, ret)
}
