package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"
	"time"

	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performSetUserVideoPrivacyProperties(router *Server, params url.Values, requestBody interface{}) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	body, err := json.Marshal(requestBody)
	So(err, ShouldBeNil)
	r, err := http.NewRequest("PUT", fmt.Sprintf("http://localhost/v1/user_video_privacy_properties/%v", params.Get("userID")), bytes.NewBuffer(body))
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)

	return w
}

func TestSetUserVideoPrivacyProperties(t *testing.T) {
	var params url.Values
	Convey("When setting props", t, func() {
		userID := int64(123)
		params = url.Values{}
		params.Set("userID", strconv.FormatInt(userID, 10))
		Convey("fail without an integer user ID", func() {
			router := mockServer()
			params.Set("userID", "test")
			body := map[string]interface{}{"hide_archives": true}
			w := performSetUserVideoPrivacyProperties(router, params, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "user_id is not a number")
		})

		Convey("fail with invalid body", func() {
			router := mockServer()
			w := performSetUserVideoPrivacyProperties(router, params, "body")
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "cannot unmarshal string")
		})

		Convey("errors when seeing unallowed fields", func() {
			router := mockServer()
			body := map[string]interface{}{"created_at": time.Now().UTC()}
			w := performSetUserVideoPrivacyProperties(router, params, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Parameter not allowed: created_at")
		})

		Convey("fail when providing non-bool option value", func() {
			router := mockServer()
			body := map[string]interface{}{"hide_archives": 100}
			w := performSetUserVideoPrivacyProperties(router, params, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid body: Unrecognized type for boolean field")
		})

		Convey("error with backend error", func() {
			router := mockServer()
			mockBackend.On("SetUserVideoPrivacyProperties", mock.Anything, userID, &models.UserVideoPrivacyPropertiesInput{}).Return(nil, fmt.Errorf("what"))
			w := performSetUserVideoPrivacyProperties(router, params, map[string]interface{}{})
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("succeed with empty options", func() {
			router := mockServer()
			mockBackend.On("SetUserVideoPrivacyProperties", mock.Anything, userID, &models.UserVideoPrivacyPropertiesInput{}).Return(nil, nil)
			w := performSetUserVideoPrivacyProperties(router, params, map[string]interface{}{})
			So(w.Code, ShouldEqual, http.StatusOK)
			So(w.Body.String(), ShouldContainSubstring, "properties")
		})

		Convey("succeed with valid options", func() {
			router := mockServer()
			mockBackend.On("SetUserVideoPrivacyProperties",
				mock.Anything,
				userID,
				&models.UserVideoPrivacyPropertiesInput{HideArchives: models.NullBool{Bool: true, Valid: true}},
			).Return(nil, nil)
			body := map[string]interface{}{"hide_archives": true}
			w := performSetUserVideoPrivacyProperties(router, params, body)
			So(w.Code, ShouldEqual, http.StatusOK)
			So(w.Body.String(), ShouldContainSubstring, "properties")
		})
	})
}
