package api

import (
	"encoding/json"
	"io/ioutil"
	"net/http"
	"strconv"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"
	"goji.io/pat"

	"golang.org/x/net/context"
)

// SetUserVODProperties sets VOD properties for a user
func (s *Server) SetUserVODProperties(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	userID := pat.Param(ctx, "user_id")
	if userID == "" {
		s.serveError(ctx, w, r, errors.MissingParamError{ParamName: "user_id"})
		return
	}
	user, err := strconv.ParseInt(userID, 10, 64)
	if err != nil {
		s.serveError(ctx, w, r, errors.IntegerParseError{ParamName: "user_id", ParamValue: userID})
		return
	}

	bodyBytes, err := ioutil.ReadAll(r.Body)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Could not read body: " + err.Error()})
		return
	}

	defer func() {
		closeErr := r.Body.Close()
		if closeErr != nil {
		}
	}()

	err = AllowBodyFields(bodyBytes, models.UserVODPropertiesFields)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	u := &models.UserVODPropertiesInput{}
	err = json.Unmarshal(bodyBytes, &u)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Invalid body: " + err.Error()})
		return
	}

	p, err := s.Backend.SetUserVODProperties(ctx, user, u)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	ret := map[string]*models.UserVODProperties{
		"properties": p,
	}
	s.serveJSON(w, r, ret)
}
