package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"
	"time"

	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performSetUserVODProperties(router *Server, params url.Values, requestBody interface{}) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	body, err := json.Marshal(requestBody)
	So(err, ShouldBeNil)
	r, err := http.NewRequest("PUT", fmt.Sprintf("http://localhost/v1/user_vod_properties/%v", params.Get("userID")), bytes.NewBuffer(body))
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)

	return w
}

func TestSetUserVODProperties(t *testing.T) {
	var params url.Values
	Convey("When setting props", t, func() {
		userID := int64(123)
		params = url.Values{}
		params.Set("userID", strconv.FormatInt(userID, 10))
		Convey("fail without an integer user ID", func() {
			router := mockServer()
			params.Set("userID", "test")
			body := map[string]interface{}{"can_upload_vod": true}
			w := performSetUserVODProperties(router, params, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "user_id is not a number")
		})

		Convey("fail with invalid body", func() {
			router := mockServer()
			w := performSetUserVODProperties(router, params, "body")
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "cannot unmarshal string")
		})

		Convey("fail when providing non-bool option value", func() {
			router := mockServer()
			body := map[string]interface{}{"can_upload_vod": 100}
			w := performSetUserVODProperties(router, params, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid body: Unrecognized type for boolean field")
		})

		Convey("errors when seeing unallowed fields", func() {
			router := mockServer()
			body := map[string]interface{}{"created_at": time.Now().UTC()}
			w := performSetUserVODProperties(router, params, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Parameter not allowed: created_at")
		})

		Convey("error if backend error", func() {
			router := mockServer()
			mockBackend.On("SetUserVODProperties", mock.Anything, userID, &models.UserVODPropertiesInput{}).Return(nil, fmt.Errorf("what"))
			w := performSetUserVODProperties(router, params, map[string]interface{}{})
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("succeed with empty options", func() {
			router := mockServer()
			mockBackend.On("SetUserVODProperties", mock.Anything, userID, &models.UserVODPropertiesInput{}).Return(nil, nil)
			w := performSetUserVODProperties(router, params, map[string]interface{}{})
			So(w.Code, ShouldEqual, http.StatusOK)
			So(w.Body.String(), ShouldContainSubstring, "properties")
		})

		Convey("succeed with valid options", func() {
			router := mockServer()
			mockBackend.On("SetUserVODProperties",
				mock.Anything,
				userID,
				&models.UserVODPropertiesInput{CanUploadVOD: models.NullBool{Bool: true, Valid: true}},
			).Return(nil, nil)
			body := map[string]interface{}{"can_upload_vod": true}
			w := performSetUserVODProperties(router, params, body)
			So(w.Code, ShouldEqual, http.StatusOK)
			So(w.Body.String(), ShouldContainSubstring, "properties")
		})
	})
}
