package api

import (
	"encoding/json"
	"io/ioutil"
	"net/http"
	"strconv"

	"code.justin.tv/vod/vinyl/errors"

	"golang.org/x/net/context"
)

// SetViewcounts sets the view counts for a set of vods.
func (s *Server) SetViewcounts(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	bodyBytes, err := ioutil.ReadAll(r.Body)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Could not read body: " + err.Error()})
		return
	}

	defer func() {
		if err := r.Body.Close(); err != nil {
		}
	}()

	in := map[string]string{}
	err = json.Unmarshal(bodyBytes, &in)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Invalid body: " + err.Error()})
		return
	}
	countsStr, ok := in["counts"]
	if !ok {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Invalid body: `counts` missing from input"})
		return
	}
	inCounts := map[string]string{}
	err = json.Unmarshal([]byte(countsStr), &inCounts)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Invalid body: `counts` was not valid JSON"})
		return
	}

	counts := map[int]int64{}
	for k, v := range inCounts {
		id, err := strconv.Atoi(k)
		if err != nil {
			s.serveError(ctx, w, r, errors.IntegerParseError{ParamName: "VOD ID", ParamValue: k})
			return
		}
		count, err := strconv.ParseInt(v, 10, 64)
		if err != nil {
			s.serveError(ctx, w, r, errors.InvalidValueError{ParamName: "count", Value: count})
			return
		}
		counts[id] = count
	}

	err = s.Backend.SetViewcounts(ctx, counts)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}
	w.WriteHeader(http.StatusNoContent)
}
