package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performSetViewcountsRequest(router *Server, requestBody interface{}) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	body, err := json.Marshal(requestBody)
	So(err, ShouldBeNil)
	r, err := http.NewRequest("PUT", "http://localhost/v1/vods/set_viewcounts", bytes.NewBuffer(body))
	So(err, ShouldBeNil)

	router.ServeHTTP(w, r)
	return w
}

func TestSetViewcounts(t *testing.T) {
	Convey("When sending reqeusts to /set_viewcounts", t, func() {
		Convey("request should fail with a json invalid body", func() {
			router := mockServer()
			bogusBody := "4head"
			w := performSetViewcountsRequest(router, bogusBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid body")
		})

		Convey("request should fail with no counts in body", func() {
			router := mockServer()
			bogusBody := map[string]string{}
			w := performSetViewcountsRequest(router, bogusBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid body")
		})

		Convey("request should fail invalid counts field", func() {
			router := mockServer()
			bogusBody := map[string]string{
				"counts": "hello",
			}
			w := performSetViewcountsRequest(router, bogusBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid body")
		})

		Convey("request should fail invalid vod id", func() {
			router := mockServer()
			bogusBody := map[string]string{
				"counts": `{"sdlj": "123"}`,
			}
			w := performSetViewcountsRequest(router, bogusBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "VOD ID")
		})

		Convey("request should fail invalid count", func() {
			router := mockServer()
			bogusBody := map[string]string{
				"counts": `{"123": "sdf"}`,
			}
			w := performSetViewcountsRequest(router, bogusBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "count")
		})

		Convey("should error for backend error", func() {
			router := mockServer()
			mockBackend.On("SetViewcounts", mock.Anything, mock.Anything).Return(fmt.Errorf("what"))
			// Make this a map[string]int64 to work
			body := map[string]string{"counts": `{"1": "322"}`}
			w := performSetViewcountsRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("should succeed with a valid body", func() {
			router := mockServer()
			mockBackend.On("SetViewcounts", mock.Anything, mock.Anything).Return(nil)
			// Make this a map[string]int64 to work
			body := map[string]string{"counts": `{"1": "322"}`}
			w := performSetViewcountsRequest(router, body)
			So(w.Code, ShouldEqual, http.StatusNoContent)
		})
	})
}
