package api

import (
	"net/http"

	"code.justin.tv/vod/vinyl/errors"

	"golang.org/x/net/context"
)

// TopMaxOffset is the max offset that the top endpoint can query for
var TopMaxOffset = 1000

var validPeriods = map[string]bool{
	"all":   true,
	"month": true,
	"week":  true,
	"day":   true,
}

var validSorts = map[string]bool{
	"time":     true,
	"views":    true,
	"trending": true,
	"trending_v3": true,
}

// Top returns most popular VOD information for a broadcast type across a set period of time.
func (s *Server) Top(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	queryValues := r.URL.Query()

	b := queryValues.Get("broadcast_type")
	if b == "" { // specifically for this endpoint, default broadcast_type to highlight
		b = "highlight"
	}
	broadcastType, err := setBroadcastType(b)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	language := commaDelimited(queryValues.Get("language"))
	game := queryValues.Get("game")

	period := queryValues.Get("period")
	if period != "" {
		if _, ok := validPeriods[period]; !ok {
			s.serveError(ctx, w, r, errors.InvalidValueError{ParamName: "period", Value: period})
			return
		}
	}

	sort := queryValues.Get("sort")
	if _, ok := validSorts[sort]; !ok {
		sort = "views"
	}

	limit, offset, err := pagination(queryValues.Get("limit"), queryValues.Get("offset"), TopMaxOffset)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	vods, err := s.Backend.Top(ctx, broadcastType, language, game, period, sort, limit, offset)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	w.Header().Add("Cache-Control", "public, max-age=900")
	s.serveJSON(w, r, vods)
}
