package api

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performTopRequest(router *Server, params url.Values) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	r, err := http.NewRequest("GET", "http://localhost/v1/vods/top", nil)
	So(err, ShouldBeNil)
	r.URL.RawQuery = params.Encode()

	router.ServeHTTP(w, r)

	return w
}

func TestTop(t *testing.T) {
	Convey("top", t, func() {
		Convey("broadcast_type", func() {
			p := url.Values{
				"period": []string{"week"},
				"limit":  []string{"20"},
				"offset": []string{"0"},
			}
			Convey("empty", func() {
				router := mockServer()
				mockBackend.On("Top", mock.Anything, []string{"highlight"}, []string{}, "", "week", "views", 20, 0).Return(nil, nil)
				w := performTopRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusOK)
				So(w.Header().Get("Cache-Control"), ShouldEqual, "public, max-age=900")
			})
			Convey("invalid", func() {
				router := mockServer()
				p.Set("broadcast_type", "bananas")
				w := performTopRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "Invalid value for broadcast_type")
				So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
			})
			Convey("valid", func() {
				router := mockServer()
				mockBackend.On("Top", mock.Anything, []string{"archive"}, []string{}, "", "week", "views", 20, 0).Return(nil, nil)
				p.Set("broadcast_type", "archive")
				w := performTopRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusOK)
				So(w.Header().Get("Cache-Control"), ShouldEqual, "public, max-age=900")
			})
			Convey("errors for backend error", func() {
				router := mockServer()
				mockBackend.On("Top", mock.Anything, []string{"archive"}, []string{}, "", "week", "views", 20, 0).Return(nil, fmt.Errorf("what"))
				p.Set("broadcast_type", "archive")
				w := performTopRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusInternalServerError)
				So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
			})
		})
		Convey("with a game", func() {
			router := mockServer()
			mockBackend.On("Top", mock.Anything, []string{"highlight"}, []string{}, "Hearthstone", "week", "views", 20, 0).Return(nil, nil)
			p := url.Values{
				"broadcast_type": []string{"highlight"},
				"game":           []string{"Hearthstone"},
				"period":         []string{"week"},
				"limit":          []string{"20"},
				"offset":         []string{"0"},
			}
			w := performTopRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusOK)
			So(w.Header().Get("Cache-Control"), ShouldEqual, "public, max-age=900")
		})

		Convey("with a sort", func() {
			router := mockServer()
			mockBackend.On("Top", mock.Anything, []string{"highlight"}, []string{}, "", "week", "trending", 20, 0).Return(nil, nil)
			p := url.Values{
				"broadcast_type": []string{"highlight"},
				"sort":           []string{"trending"},
				"period":         []string{"week"},
				"limit":          []string{"20"},
				"offset":         []string{"0"},
			}
			w := performTopRequest(router, p)
			So(w.Code, ShouldEqual, http.StatusOK)
			So(w.Header().Get("Cache-Control"), ShouldEqual, "public, max-age=900")
		})
		Convey("period", func() {
			p := url.Values{
				"broadcast_type": []string{"highlight"},
				"limit":          []string{"20"},
				"offset":         []string{"0"},
			}
			Convey("valid", func() {
				router := mockServer()
				mockBackend.On("Top", mock.Anything, []string{"highlight"}, []string{}, "", "all", "views", 20, 0).Return(nil, nil)
				p.Set("period", "all")
				w := performTopRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusOK)
				So(w.Header().Get("Cache-Control"), ShouldEqual, "public, max-age=900")
			})
			Convey("empty is ok", func() {
				router := mockServer()
				mockBackend.On("Top", mock.Anything, []string{"highlight"}, []string{}, "", "", "views", 20, 0).Return(nil, nil)
				p.Set("period", "")
				w := performTopRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusOK)
				So(w.Header().Get("Cache-Control"), ShouldEqual, "public, max-age=900")
			})
			Convey("invalid same as empty", func() {
				router := mockServer()
				p.Set("period", "bananas")
				w := performTopRequest(router, p)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
				So(w.Body.String(), ShouldContainSubstring, "Invalid value for period")
				So(w.Header().Get("Cache-Control"), ShouldBeEmpty)
			})
		})
	})
}
