package api

import (
	"encoding/json"
	"io/ioutil"
	"net/http"
	"strconv"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"
	"code.justin.tv/vod/vinyl/utils"
	"goji.io/pat"
	"golang.org/x/net/context"
)

// UpdateAudibleMagicResponses supports update an audible magic response entry.
func (s *Server) UpdateAudibleMagicResponses(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	amr := pat.Param(ctx, "amr_id")
	if amr == "" {
		s.serveError(ctx, w, r, errors.MissingParamError{ParamName: "amr_id"})
		return
	}
	amrID, err := strconv.Atoi(amr)
	if err != nil {
		s.serveError(ctx, w, r, errors.IntegerParseError{ParamName: "amr_id", ParamValue: amr})
		return
	}

	bodyBytes, err := ioutil.ReadAll(r.Body)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Could not read body: " + err.Error()})
		return
	}

	defer func() {
		closeErr := r.Body.Close()
		if closeErr != nil {
		}
	}()

	err = AllowBodyFields(bodyBytes, models.AMRUpdateFields)
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	amrObj := models.AMR{}
	err = json.Unmarshal(bodyBytes, &amrObj)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Invalid body: " + err.Error()})
		return
	}

	updateFields := map[string]interface{}{}
	dec := utils.JSONDecoderWithNumbers(bodyBytes)
	err = dec.Decode(&updateFields)
	if err != nil {
		s.serveError(ctx, w, r, errors.RequestBodyError{ErrorString: "Invalid body: " + err.Error()})
		return
	}
	updatedAMR, err := s.Backend.UpdateAMR(ctx, amrID, utils.NormalizeJSONMap(updateFields))
	if err != nil {
		s.serveError(ctx, w, r, err)
		return
	}

	ret := map[string]*models.AMR{
		"audible_magic_response": updatedAMR,
	}

	s.serveJSON(w, r, ret)
}
